"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardsAutoInstaller = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const message_service_1 = require("@theia/core/lib/common/message-service");
const boards_service_1 = require("../../common/protocol/boards-service");
const boards_service_provider_1 = require("./boards-service-provider");
const protocol_1 = require("../../common/protocol");
const boards_widget_frontend_contribution_1 = require("./boards-widget-frontend-contribution");
const common_1 = require("@theia/core/lib/common");
const notification_center_1 = require("../notification-center");
const nls_1 = require("../../common/nls");
/**
 * Listens on `BoardsConfig.Config` changes, if a board is selected which does not
 * have the corresponding core installed, it proposes the user to install the core.
 */
// * Cases in which we do not show the auto-install prompt:
// 1. When a related platform is already installed
// 2. When a prompt is already showing in the UI
// 3. When a board is unplugged
let BoardsAutoInstaller = class BoardsAutoInstaller {
    constructor() {
        // Workaround for https://github.com/eclipse-theia/theia/issues/9349
        this.notifications = [];
    }
    onStart() {
        const setEventListeners = () => {
            this.boardsServiceClient.onBoardsConfigChanged((config) => {
                const { selectedBoard, selectedPort } = config;
                const boardWasUnplugged = !selectedPort && this.portSelectedOnLastRefusal;
                this.clearLastRefusedPromptInfo();
                if (boardWasUnplugged ||
                    !selectedBoard ||
                    this.promptAlreadyShowingForBoard(selectedBoard)) {
                    return;
                }
                this.ensureCoreExists(selectedBoard, selectedPort);
            });
            // we "clearRefusedPackageInfo" if a "refused" package is eventually
            // installed, though this is not strictly necessary. It's more of a
            // cleanup, to ensure the related variables are representative of
            // current state.
            this.notificationCenter.onPlatformDidInstall((installed) => {
                if (this.lastRefusedPackageId === installed.item.id) {
                    this.clearLastRefusedPromptInfo();
                }
            });
        };
        // we should invoke this.ensureCoreExists only once we're sure
        // everything has been reconciled
        this.boardsServiceClient.reconciled.then(() => {
            const { selectedBoard, selectedPort } = this.boardsServiceClient.boardsConfig;
            if (selectedBoard) {
                this.ensureCoreExists(selectedBoard, selectedPort);
            }
            setEventListeners();
        });
    }
    removeNotificationByBoard(selectedBoard) {
        const index = this.notifications.findIndex((notification) => boards_service_1.Board.sameAs(notification, selectedBoard));
        if (index !== -1) {
            this.notifications.splice(index, 1);
        }
    }
    clearLastRefusedPromptInfo() {
        this.lastRefusedPackageId = undefined;
        this.portSelectedOnLastRefusal = undefined;
    }
    setLastRefusedPromptInfo(packageId, selectedPort) {
        this.lastRefusedPackageId = packageId;
        this.portSelectedOnLastRefusal = selectedPort;
    }
    promptAlreadyShowingForBoard(board) {
        return Boolean(this.notifications.find((notification) => boards_service_1.Board.sameAs(notification, board)));
    }
    ensureCoreExists(selectedBoard, selectedPort) {
        this.notifications.push(selectedBoard);
        this.boardsService.search({}).then((packages) => {
            const candidate = this.getInstallCandidate(packages, selectedBoard);
            if (candidate) {
                this.showAutoInstallPrompt(candidate, selectedBoard, selectedPort);
            }
            else {
                this.removeNotificationByBoard(selectedBoard);
            }
        });
    }
    getInstallCandidate(packages, selectedBoard) {
        // filter packagesForBoard selecting matches from the cli (installed packages)
        // and matches based on the board name
        // NOTE: this ensures the Deprecated & new packages are all in the array
        // so that we can check if any of the valid packages is already installed
        const packagesForBoard = packages.filter((pkg) => boards_service_1.BoardsPackage.contains(selectedBoard, pkg) ||
            pkg.boards.some((board) => board.name === selectedBoard.name));
        // check if one of the packages for the board is already installed. if so, no hint
        if (packagesForBoard.some(({ installedVersion }) => !!installedVersion)) {
            return;
        }
        // filter the installable (not installed) packages,
        // CLI returns the packages already sorted with the deprecated ones at the end of the list
        // in order to ensure the new ones are preferred
        const candidates = packagesForBoard.filter(({ installable, installedVersion }) => installable && !installedVersion);
        return candidates[0];
    }
    showAutoInstallPrompt(candidate, selectedBoard, selectedPort) {
        const candidateName = candidate.name;
        const version = candidate.availableVersions[0]
            ? `[v ${candidate.availableVersions[0]}]`
            : '';
        const info = this.generatePromptInfoText(candidateName, version, selectedBoard.name);
        const actions = this.createPromptActions(candidate);
        const onRefuse = () => {
            this.setLastRefusedPromptInfo(candidate.id, selectedPort);
        };
        const handleAction = this.createOnAnswerHandler(actions, onRefuse);
        const onAnswer = (answer) => {
            this.removeNotificationByBoard(selectedBoard);
            handleAction(answer);
        };
        this.messageService
            .info(info, ...actions.map((action) => action.key))
            .then(onAnswer);
    }
    generatePromptInfoText(candidateName, version, boardName) {
        return common_1.nls.localize('arduino/board/installNow', 'The "{0} {1}" core has to be installed for the currently selected "{2}" board. Do you want to install it now?', candidateName, version, boardName);
    }
    createPromptActions(candidate) {
        const yes = common_1.nls.localize('vscode/extensionsUtils/yes', 'Yes');
        const actions = [
            {
                key: nls_1.InstallManually,
                handler: () => {
                    this.boardsManagerFrontendContribution
                        .openView({ reveal: true })
                        .then((widget) => widget.refresh({
                        query: candidate.name.toLocaleLowerCase(),
                        type: 'All',
                    }));
                },
            },
            {
                isAcceptance: true,
                key: yes,
                handler: () => {
                    return protocol_1.Installable.installWithProgress({
                        installable: this.boardsService,
                        item: candidate,
                        messageService: this.messageService,
                        responseService: this.responseService,
                        version: candidate.availableVersions[0],
                    });
                },
            },
        ];
        return actions;
    }
    createOnAnswerHandler(actions, onRefuse) {
        return (answer) => {
            const actionToHandle = actions.find((action) => action.key === answer);
            actionToHandle === null || actionToHandle === void 0 ? void 0 : actionToHandle.handler();
            if (!(actionToHandle === null || actionToHandle === void 0 ? void 0 : actionToHandle.isAcceptance) && onRefuse) {
                onRefuse();
            }
        };
    }
};
__decorate([
    (0, inversify_1.inject)(notification_center_1.NotificationCenter),
    __metadata("design:type", notification_center_1.NotificationCenter)
], BoardsAutoInstaller.prototype, "notificationCenter", void 0);
__decorate([
    (0, inversify_1.inject)(message_service_1.MessageService),
    __metadata("design:type", message_service_1.MessageService)
], BoardsAutoInstaller.prototype, "messageService", void 0);
__decorate([
    (0, inversify_1.inject)(boards_service_1.BoardsService),
    __metadata("design:type", Object)
], BoardsAutoInstaller.prototype, "boardsService", void 0);
__decorate([
    (0, inversify_1.inject)(boards_service_provider_1.BoardsServiceProvider),
    __metadata("design:type", boards_service_provider_1.BoardsServiceProvider)
], BoardsAutoInstaller.prototype, "boardsServiceClient", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.ResponseServiceClient),
    __metadata("design:type", Object)
], BoardsAutoInstaller.prototype, "responseService", void 0);
__decorate([
    (0, inversify_1.inject)(boards_widget_frontend_contribution_1.BoardsListWidgetFrontendContribution),
    __metadata("design:type", boards_widget_frontend_contribution_1.BoardsListWidgetFrontendContribution)
], BoardsAutoInstaller.prototype, "boardsManagerFrontendContribution", void 0);
BoardsAutoInstaller = __decorate([
    (0, inversify_1.injectable)()
], BoardsAutoInstaller);
exports.BoardsAutoInstaller = BoardsAutoInstaller;
//# sourceMappingURL=boards-auto-installer.js.map