"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardsConfig = exports.Item = void 0;
const React = require("@theia/core/shared/react");
const objects_1 = require("@theia/core/lib/common/objects");
const disposable_1 = require("@theia/core/lib/common/disposable");
const boards_service_1 = require("../../common/protocol/boards-service");
const utils_1 = require("../../common/utils");
const common_1 = require("@theia/core/lib/common");
class Item extends React.Component {
    constructor() {
        super(...arguments);
        this.onClick = () => {
            this.props.onClick(this.props.item);
        };
    }
    render() {
        const { selected, label, missing, details } = this.props;
        const classNames = ['item'];
        if (selected) {
            classNames.push('selected');
        }
        if (missing === true) {
            classNames.push('missing');
        }
        return (React.createElement("div", { onClick: this.onClick, className: classNames.join(' '), title: `${label}${!details ? '' : details}` },
            React.createElement("div", { className: "label" }, label),
            !details ? '' : React.createElement("div", { className: "details" }, details),
            !selected ? ('') : (React.createElement("div", { className: "selected-icon" },
                React.createElement("i", { className: "fa fa-check" })))));
    }
}
exports.Item = Item;
class BoardsConfig extends React.Component {
    constructor(props) {
        super(props);
        this.toDispose = new disposable_1.DisposableCollection();
        this.updateBoards = (eventOrQuery = '') => {
            const query = typeof eventOrQuery === 'string'
                ? eventOrQuery
                : eventOrQuery.target.value.toLowerCase();
            this.setState({ query });
            this.queryBoards({ query }).then((searchResults) => this.setState({ searchResults }));
        };
        this.updatePorts = (ports = [], removedPorts = []) => {
            this.queryPorts(Promise.resolve(ports)).then(({ knownPorts }) => {
                let { selectedPort } = this.state;
                // If the currently selected port is not available anymore, unset the selected port.
                if (removedPorts.some((port) => boards_service_1.Port.sameAs(port, selectedPort))) {
                    selectedPort = undefined;
                }
                this.setState({ knownPorts, selectedPort }, () => this.fireConfigChanged());
            });
        };
        this.queryBoards = (options = {}) => {
            return this.props.boardsServiceProvider.searchBoards(options);
        };
        this.queryPorts = async (availablePorts = this.availablePorts) => {
            // Available ports must be sorted in this order:
            // 1. Serial with recognized boards
            // 2. Serial with guessed boards
            // 3. Serial with incomplete boards
            // 4. Network with recognized boards
            // 5. Other protocols with recognized boards
            const ports = (await availablePorts).sort((left, right) => {
                if (left.protocol === 'serial' && right.protocol !== 'serial') {
                    return -1;
                }
                else if (left.protocol !== 'serial' && right.protocol === 'serial') {
                    return 1;
                }
                else if (left.protocol === 'network' && right.protocol !== 'network') {
                    return -1;
                }
                else if (left.protocol !== 'network' && right.protocol === 'network') {
                    return 1;
                }
                else if (left.protocol === right.protocol) {
                    // We show ports, including those that have guessed
                    // or unrecognized boards, so we must sort those too.
                    const leftBoard = this.availableBoards.find((board) => board.port === left);
                    const rightBoard = this.availableBoards.find((board) => board.port === right);
                    if (leftBoard && !rightBoard) {
                        return -1;
                    }
                    else if (!leftBoard && rightBoard) {
                        return 1;
                    }
                    else if ((leftBoard === null || leftBoard === void 0 ? void 0 : leftBoard.state) < (rightBoard === null || rightBoard === void 0 ? void 0 : rightBoard.state)) {
                        return -1;
                    }
                    else if ((leftBoard === null || leftBoard === void 0 ? void 0 : leftBoard.state) > (rightBoard === null || rightBoard === void 0 ? void 0 : rightBoard.state)) {
                        return 1;
                    }
                }
                return (0, utils_1.naturalCompare)(left.address, right.address);
            });
            return { knownPorts: ports };
        };
        this.toggleFilterPorts = () => {
            this.setState({ showAllPorts: !this.state.showAllPorts });
        };
        this.selectPort = (selectedPort) => {
            this.setState({ selectedPort }, () => this.fireConfigChanged());
        };
        this.selectBoard = (selectedBoard) => {
            this.setState({ selectedBoard }, () => this.fireConfigChanged());
        };
        this.focusNodeSet = (element) => {
            this.props.onFocusNodeSet(element || undefined);
        };
        const { boardsConfig } = props.boardsServiceProvider;
        this.state = Object.assign({ searchResults: [], knownPorts: [], showAllPorts: false, query: '' }, boardsConfig);
    }
    componentDidMount() {
        this.toDispose.pushAll([
            this.props.onAppStateDidChange((state) => {
                if (state === 'ready') {
                    this.updateBoards();
                    this.updatePorts(this.props.boardsServiceProvider.availableBoards
                        .map(({ port }) => port)
                        .filter(objects_1.notEmpty));
                }
            }),
            this.props.notificationCenter.onAttachedBoardsDidChange((event) => this.updatePorts(event.newState.ports, boards_service_1.AttachedBoardsChangeEvent.diff(event).detached.ports)),
            this.props.boardsServiceProvider.onBoardsConfigChanged(({ selectedBoard, selectedPort }) => {
                this.setState({ selectedBoard, selectedPort }, () => this.fireConfigChanged());
            }),
            this.props.notificationCenter.onPlatformDidInstall(() => this.updateBoards(this.state.query)),
            this.props.notificationCenter.onPlatformDidUninstall(() => this.updateBoards(this.state.query)),
            this.props.notificationCenter.onIndexDidUpdate(() => this.updateBoards(this.state.query)),
            this.props.notificationCenter.onDaemonDidStart(() => this.updateBoards(this.state.query)),
            this.props.notificationCenter.onDaemonDidStop(() => this.setState({ searchResults: [] })),
            this.props.onFilteredTextDidChangeEvent((query) => this.setState({ query }, () => this.updateBoards(this.state.query))),
        ]);
    }
    componentWillUnmount() {
        this.toDispose.dispose();
    }
    fireConfigChanged() {
        const { selectedBoard, selectedPort } = this.state;
        this.props.onConfigChange({ selectedBoard, selectedPort });
    }
    get availablePorts() {
        return this.props.boardsServiceProvider.availableBoards
            .map(({ port }) => port)
            .filter(objects_1.notEmpty);
    }
    get availableBoards() {
        return this.props.boardsServiceProvider.availableBoards;
    }
    render() {
        return (React.createElement(React.Fragment, null,
            this.renderContainer('boards', this.renderBoards.bind(this)),
            this.renderContainer('ports', this.renderPorts.bind(this), this.renderPortsFooter.bind(this))));
    }
    renderContainer(title, contentRenderer, footerRenderer) {
        return (React.createElement("div", { className: "container" },
            React.createElement("div", { className: "content" },
                React.createElement("div", { className: "title" }, title),
                contentRenderer(),
                React.createElement("div", { className: "footer" }, footerRenderer ? footerRenderer() : ''))));
    }
    renderBoards() {
        const { selectedBoard, searchResults, query } = this.state;
        // Board names are not unique per core https://github.com/arduino/arduino-pro-ide/issues/262#issuecomment-661019560
        // It is tricky when the core is not yet installed, no FQBNs are available.
        const distinctBoards = new Map();
        const toKey = ({ name, packageName, fqbn }) => !!fqbn ? `${name}-${packageName}-${fqbn}` : `${name}-${packageName}`;
        for (const board of boards_service_1.Board.decorateBoards(selectedBoard, searchResults)) {
            const key = toKey(board);
            if (!distinctBoards.has(key)) {
                distinctBoards.set(key, board);
            }
        }
        return (React.createElement(React.Fragment, null,
            React.createElement("div", { className: "search" },
                React.createElement("input", { type: "search", value: query, className: "theia-input", placeholder: common_1.nls.localize('arduino/board/searchBoard', 'Search board'), onChange: this.updateBoards, ref: this.focusNodeSet }),
                React.createElement("i", { className: "fa fa-search" })),
            React.createElement("div", { className: "boards list" }, Array.from(distinctBoards.values()).map((board) => (React.createElement(Item, { key: toKey(board), item: board, label: board.name, details: board.details, selected: board.selected, onClick: this.selectBoard, missing: board.missing }))))));
    }
    renderPorts() {
        let ports = [];
        if (this.state.showAllPorts) {
            ports = this.state.knownPorts;
        }
        else {
            ports = this.state.knownPorts.filter(boards_service_1.Port.visiblePorts(this.availableBoards));
        }
        return !ports.length ? (React.createElement("div", { className: "loading noselect" }, common_1.nls.localize('arduino/board/noPortsDiscovered', 'No ports discovered'))) : (React.createElement("div", { className: "ports list" }, ports.map((port) => (React.createElement(Item, { key: `${boards_service_1.Port.keyOf(port)}`, item: port, label: boards_service_1.Port.toString(port), selected: boards_service_1.Port.sameAs(this.state.selectedPort, port), onClick: this.selectPort })))));
    }
    renderPortsFooter() {
        return (React.createElement("div", { className: "noselect" },
            React.createElement("label", { title: common_1.nls.localize('arduino/board/showAllAvailablePorts', 'Shows all available ports when enabled') },
                React.createElement("input", { type: "checkbox", defaultChecked: this.state.showAllPorts, onChange: this.toggleFilterPorts }),
                React.createElement("span", null, "Show all ports"))));
    }
}
exports.BoardsConfig = BoardsConfig;
(function (BoardsConfig) {
    let Config;
    (function (Config) {
        function sameAs(config, other) {
            const { selectedBoard, selectedPort } = config;
            if (boards_service_1.Board.is(other)) {
                return (!!selectedBoard &&
                    boards_service_1.Board.equals(other, selectedBoard) &&
                    boards_service_1.Port.sameAs(selectedPort, other.port));
            }
            return sameAs(config, other);
        }
        Config.sameAs = sameAs;
        function equals(left, right) {
            return (left.selectedBoard === right.selectedBoard &&
                left.selectedPort === right.selectedPort);
        }
        Config.equals = equals;
        function toString(config, options = { default: '' }) {
            const { selectedBoard, selectedPort: port } = config;
            if (!selectedBoard) {
                return options.default;
            }
            const { name } = selectedBoard;
            return `${name}${port ? ` at ${port.address}` : ''}`;
        }
        Config.toString = toString;
        function setConfig(config, urlToAttachTo) {
            const copy = new URL(urlToAttachTo.toString());
            if (!config) {
                copy.searchParams.delete('boards-config');
                return copy;
            }
            const selectedBoard = config.selectedBoard
                ? {
                    name: config.selectedBoard.name,
                    fqbn: config.selectedBoard.fqbn,
                }
                : undefined;
            const selectedPort = config.selectedPort
                ? {
                    protocol: config.selectedPort.protocol,
                    address: config.selectedPort.address,
                }
                : undefined;
            const jsonConfig = JSON.stringify({ selectedBoard, selectedPort });
            copy.searchParams.set('boards-config', encodeURIComponent(jsonConfig));
            return copy;
        }
        Config.setConfig = setConfig;
        function getConfig(url) {
            const encoded = url.searchParams.get('boards-config');
            if (!encoded) {
                return undefined;
            }
            try {
                const raw = decodeURIComponent(encoded);
                const candidate = JSON.parse(raw);
                if (typeof candidate === 'object') {
                    return candidate;
                }
                console.warn(`Expected candidate to be an object. It was ${typeof candidate}. URL was: ${url}`);
                return undefined;
            }
            catch (e) {
                console.log(`Could not get board config from URL: ${url}.`, e);
                return undefined;
            }
        }
        Config.getConfig = getConfig;
    })(Config = BoardsConfig.Config || (BoardsConfig.Config = {}));
})(BoardsConfig = exports.BoardsConfig || (exports.BoardsConfig = {}));
//# sourceMappingURL=boards-config.js.map