"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AvailableBoard = exports.BoardsServiceProvider = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const event_1 = require("@theia/core/lib/common/event");
const logger_1 = require("@theia/core/lib/common/logger");
const command_1 = require("@theia/core/lib/common/command");
const message_service_1 = require("@theia/core/lib/common/message-service");
const protocol_1 = require("../../common/protocol");
const boards_config_1 = require("./boards-config");
const utils_1 = require("../../common/utils");
const notification_center_1 = require("../notification-center");
const storage_wrapper_1 = require("../storage-wrapper");
const common_1 = require("@theia/core/lib/common");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const frontend_application_state_1 = require("@theia/core/lib/browser/frontend-application-state");
const nls_1 = require("../../common/nls");
let BoardsServiceProvider = class BoardsServiceProvider {
    constructor() {
        this.onBoardsConfigChangedEmitter = new event_1.Emitter();
        this.onAvailableBoardsChangedEmitter = new event_1.Emitter();
        this.onAvailablePortsChangedEmitter = new event_1.Emitter();
        /**
         * Used for the auto-reconnecting. Sometimes, the attached board gets disconnected after uploading something to it.
         * It happens with certain boards on Windows. For example, the `MKR1000` boards is selected on post `COM5` on Windows,
         * perform an upload, the board automatically disconnects and reconnects, but on another port, `COM10`.
         * We have to listen on such changes and auto-reconnect the same board on another port.
         * See: https://arduino.slack.com/archives/CJJHJCJSJ/p1568645417013000?thread_ts=1568640504.009400&cid=CJJHJCJSJ
         */
        this.latestValidBoardsConfig = undefined;
        this.latestBoardsConfig = undefined;
        this._boardsConfig = {};
        this._attachedBoards = []; // This does not contain the `Unknown` boards. They're visible from the available ports only.
        this._availablePorts = [];
        this._availableBoards = [];
        /**
         * Unlike `onAttachedBoardsChanged` this event fires when the user modifies the selected board in the IDE.\
         * This event also fires, when the boards package was not available for the currently selected board,
         * and the user installs the board package. Note: installing a board package will set the `fqbn` of the
         * currently selected board.
         *
         * This event is also emitted when the board package for the currently selected board was uninstalled.
         */
        this.onBoardsConfigChanged = this.onBoardsConfigChangedEmitter.event;
        this.onAvailableBoardsChanged = this.onAvailableBoardsChangedEmitter.event;
        this.onAvailablePortsChanged = this.onAvailablePortsChangedEmitter.event;
        this._reconciled = new promise_util_1.Deferred();
    }
    onStart() {
        this.notificationCenter.onAttachedBoardsDidChange(this.notifyAttachedBoardsChanged.bind(this));
        this.notificationCenter.onPlatformDidInstall(this.notifyPlatformInstalled.bind(this));
        this.notificationCenter.onPlatformDidUninstall(this.notifyPlatformUninstalled.bind(this));
        this.appStateService.reachedState('ready').then(async () => {
            const [state] = await Promise.all([
                this.boardsService.getState(),
                this.loadState(),
            ]);
            const { boards: attachedBoards, ports: availablePorts } = protocol_1.AvailablePorts.split(state);
            this._attachedBoards = attachedBoards;
            this._availablePorts = availablePorts;
            this.onAvailablePortsChangedEmitter.fire(this._availablePorts);
            await this.reconcileAvailableBoards();
            this.tryReconnect();
            this._reconciled.resolve();
        });
    }
    get reconciled() {
        return this._reconciled.promise;
    }
    snapshotBoardDiscoveryOnUpload() {
        this.lastBoardsConfigOnUpload = this._boardsConfig;
        this.lastAvailablePortsOnUpload = this._availablePorts;
    }
    clearBoardDiscoverySnapshot() {
        this.lastBoardsConfigOnUpload = undefined;
        this.lastAvailablePortsOnUpload = undefined;
    }
    portToAutoSelectCanBeDerived() {
        return Boolean(this.lastBoardsConfigOnUpload && this.lastAvailablePortsOnUpload);
    }
    attemptPostUploadAutoSelect() {
        setTimeout(() => {
            if (this.portToAutoSelectCanBeDerived()) {
                this.attemptAutoSelect({
                    ports: this._availablePorts,
                    boards: this._availableBoards,
                });
            }
        }, 2000); // 2 second delay same as IDE 1.8
    }
    attemptAutoSelect(newState) {
        this.deriveBoardConfigToAutoSelect(newState);
        this.tryReconnect();
    }
    deriveBoardConfigToAutoSelect(newState) {
        if (!this.portToAutoSelectCanBeDerived()) {
            this.boardConfigToAutoSelect = undefined;
            return;
        }
        const oldPorts = this.lastAvailablePortsOnUpload;
        const { ports: newPorts, boards: newBoards } = newState;
        const appearedPorts = oldPorts.length > 0
            ? newPorts.filter((newPort) => oldPorts.every((oldPort) => !protocol_1.Port.sameAs(newPort, oldPort)))
            : newPorts;
        for (const port of appearedPorts) {
            const boardOnAppearedPort = newBoards.find((board) => protocol_1.Port.sameAs(board.port, port));
            const lastBoardsConfigOnUpload = this.lastBoardsConfigOnUpload;
            if (boardOnAppearedPort &&
                lastBoardsConfigOnUpload.selectedBoard &&
                protocol_1.Board.sameAs(boardOnAppearedPort, lastBoardsConfigOnUpload.selectedBoard)) {
                this.clearBoardDiscoverySnapshot();
                this.boardConfigToAutoSelect = {
                    selectedBoard: boardOnAppearedPort,
                    selectedPort: port,
                };
                return;
            }
        }
    }
    notifyAttachedBoardsChanged(event) {
        if (!protocol_1.AttachedBoardsChangeEvent.isEmpty(event)) {
            this.logger.info('Attached boards and available ports changed:');
            this.logger.info(protocol_1.AttachedBoardsChangeEvent.toString(event));
            this.logger.info('------------------------------------------');
        }
        this._attachedBoards = event.newState.boards;
        this._availablePorts = event.newState.ports;
        this.onAvailablePortsChangedEmitter.fire(this._availablePorts);
        this.reconcileAvailableBoards().then(() => {
            const { uploadInProgress } = event;
            // avoid attempting "auto-selection" while an
            // upload is in progress
            if (!uploadInProgress) {
                this.attemptAutoSelect(event.newState);
            }
        });
    }
    notifyPlatformInstalled(event) {
        this.logger.info('Boards package installed: ', JSON.stringify(event));
        const { selectedBoard } = this.boardsConfig;
        const { installedVersion, id } = event.item;
        if (selectedBoard) {
            const installedBoard = event.item.boards.find(({ name }) => name === selectedBoard.name);
            if (installedBoard &&
                (!selectedBoard.fqbn || selectedBoard.fqbn === installedBoard.fqbn)) {
                this.logger.info(`Board package ${id}[${installedVersion}] was installed. Updating the FQBN of the currently selected ${selectedBoard.name} board. [FQBN: ${installedBoard.fqbn}]`);
                this.boardsConfig = Object.assign(Object.assign({}, this.boardsConfig), { selectedBoard: installedBoard });
                return;
            }
            // The board name can change after install.
            // This logic handles it "gracefully" by unselecting the board, so that we can avoid no FQBN is set error.
            // https://github.com/arduino/arduino-cli/issues/620
            // https://github.com/arduino/arduino-pro-ide/issues/374
            if (protocol_1.BoardWithPackage.is(selectedBoard) &&
                selectedBoard.packageId === event.item.id &&
                !installedBoard) {
                const yes = common_1.nls.localize('vscode/extensionsUtils/yes', 'Yes');
                this.messageService
                    .warn(common_1.nls.localize('arduino/board/couldNotFindPreviouslySelected', "Could not find previously selected board '{0}' in installed platform '{1}'. Please manually reselect the board you want to use. Do you want to reselect it now?", selectedBoard.name, event.item.name), common_1.nls.localize('arduino/board/reselectLater', 'Reselect later'), yes)
                    .then(async (answer) => {
                    if (answer === yes) {
                        this.commandService.executeCommand('arduino-open-boards-dialog', selectedBoard.name);
                    }
                });
                this.boardsConfig = {};
                return;
            }
            // Trigger a board re-set. See: https://github.com/arduino/arduino-cli/issues/954
            // E.g: install `adafruit:avr`, then select `adafruit:avr:adafruit32u4` board, and finally install the required `arduino:avr`
            this.boardsConfig = this.boardsConfig;
        }
    }
    notifyPlatformUninstalled(event) {
        this.logger.info('Boards package uninstalled: ', JSON.stringify(event));
        const { selectedBoard } = this.boardsConfig;
        if (selectedBoard && selectedBoard.fqbn) {
            const uninstalledBoard = event.item.boards.find(({ name }) => name === selectedBoard.name);
            if (uninstalledBoard && uninstalledBoard.fqbn === selectedBoard.fqbn) {
                // We should not unset the FQBN, if the selected board is an attached, recognized board.
                // Attach Uno and install AVR, select Uno. Uninstall the AVR core while Uno is selected. We do not want to discard the FQBN of the Uno board.
                // Dev note: We cannot assume the `selectedBoard` is a type of `AvailableBoard`.
                // When the user selects an `AvailableBoard` it works, but between app start/stops,
                // it is just a FQBN, so we need to find the `selected` board among the `AvailableBoards`
                const selectedAvailableBoard = AvailableBoard.is(selectedBoard)
                    ? selectedBoard
                    : this._availableBoards.find((availableBoard) => protocol_1.Board.sameAs(availableBoard, selectedBoard));
                if (selectedAvailableBoard &&
                    selectedAvailableBoard.selected &&
                    selectedAvailableBoard.state === AvailableBoard.State.recognized) {
                    return;
                }
                this.logger.info(`Board package ${event.item.id} was uninstalled. Discarding the FQBN of the currently selected ${selectedBoard.name} board.`);
                const selectedBoardWithoutFqbn = {
                    name: selectedBoard.name,
                    // No FQBN
                };
                this.boardsConfig = Object.assign(Object.assign({}, this.boardsConfig), { selectedBoard: selectedBoardWithoutFqbn });
            }
        }
    }
    tryReconnect() {
        if (this.latestValidBoardsConfig && !this.canUploadTo(this.boardsConfig)) {
            for (const board of this.availableBoards.filter(({ state }) => state !== AvailableBoard.State.incomplete)) {
                if (this.latestValidBoardsConfig.selectedBoard.fqbn === board.fqbn &&
                    this.latestValidBoardsConfig.selectedBoard.name === board.name &&
                    protocol_1.Port.sameAs(this.latestValidBoardsConfig.selectedPort, board.port)) {
                    this.boardsConfig = this.latestValidBoardsConfig;
                    return true;
                }
            }
            if (!this.boardConfigToAutoSelect)
                return false;
            this.boardsConfig = this.boardConfigToAutoSelect;
            this.boardConfigToAutoSelect = undefined;
            return true;
        }
        return false;
    }
    set boardsConfig(config) {
        this.setBoardsConfig(config);
        this.saveState().finally(() => this.reconcileAvailableBoards().finally(() => this.onBoardsConfigChangedEmitter.fire(this._boardsConfig)));
    }
    get boardsConfig() {
        return this._boardsConfig;
    }
    setBoardsConfig(config) {
        this.logger.debug('Board config changed: ', JSON.stringify(config));
        this._boardsConfig = config;
        this.latestBoardsConfig = this._boardsConfig;
        if (this.canUploadTo(this._boardsConfig)) {
            this.latestValidBoardsConfig = this._boardsConfig;
        }
    }
    async searchBoards({ query, cores, }) {
        const boards = await this.boardsService.searchBoards({ query });
        return boards;
    }
    async selectedBoardUserFields() {
        if (!this._boardsConfig.selectedBoard || !this._boardsConfig.selectedPort) {
            return [];
        }
        const fqbn = this._boardsConfig.selectedBoard.fqbn;
        if (!fqbn) {
            return [];
        }
        const protocol = this._boardsConfig.selectedPort.protocol;
        return await this.boardsService.getBoardUserFields({ fqbn, protocol });
    }
    /**
     * `true` if the `config.selectedBoard` is defined; hence can compile against the board. Otherwise, `false`.
     */
    canVerify(config = this.boardsConfig, options = { silent: true }) {
        if (!config) {
            return false;
        }
        if (!config.selectedBoard) {
            if (!options.silent) {
                this.messageService.warn(common_1.nls.localize('arduino/board/noneSelected', 'No boards selected.'), {
                    timeout: 3000,
                });
            }
            return false;
        }
        return true;
    }
    /**
     * `true` if `canVerify`, the board has an FQBN and the `config.selectedPort` is also set, hence can upload to board. Otherwise, `false`.
     */
    canUploadTo(config = this.boardsConfig, options = { silent: true }) {
        if (!this.canVerify(config, options)) {
            return false;
        }
        const { name } = config.selectedBoard;
        if (!config.selectedPort) {
            if (!options.silent) {
                this.messageService.warn(common_1.nls.localize('arduino/board/noPortsSelected', "No ports selected for board: '{0}'.", name), {
                    timeout: 3000,
                });
            }
            return false;
        }
        if (!config.selectedBoard.fqbn) {
            if (!options.silent) {
                this.messageService.warn(common_1.nls.localize('arduino/board/noFQBN', 'The FQBN is not available for the selected board "{0}". Do you have the corresponding core installed?', name), { timeout: 3000 });
            }
            return false;
        }
        return true;
    }
    get availableBoards() {
        return this._availableBoards;
    }
    /**
     * @deprecated Do not use this API, it will be removed. This is a hack to be able to set the missing port `properties` before an upload.
     *
     * See: https://github.com/arduino/arduino-ide/pull/1335#issuecomment-1224355236.
     */
    // TODO: remove this API and fix the selected board config store/restore correctly.
    get availablePorts() {
        return this._availablePorts.slice();
    }
    async waitUntilAvailable(what, timeout) {
        const find = (needle, haystack) => haystack.find((board) => protocol_1.Board.equals(needle, board) && protocol_1.Port.sameAs(needle.port, board.port));
        const timeoutTask = !!timeout && timeout > 0
            ? new Promise((_, reject) => setTimeout(() => reject(new Error(`Timeout after ${timeout} ms.`)), timeout))
            : new Promise(() => {
                /* never */
            });
        const waitUntilTask = new Promise((resolve) => {
            let candidate = find(what, this.availableBoards);
            if (candidate) {
                resolve();
                return;
            }
            const disposable = this.onAvailableBoardsChanged((availableBoards) => {
                candidate = find(what, availableBoards);
                if (candidate) {
                    disposable.dispose();
                    resolve();
                }
            });
        });
        return await Promise.race([waitUntilTask, timeoutTask]);
    }
    async reconcileAvailableBoards() {
        const availablePorts = this._availablePorts;
        // Unset the port on the user's config, if it is not available anymore.
        if (this.boardsConfig.selectedPort &&
            !availablePorts.some((port) => protocol_1.Port.sameAs(port, this.boardsConfig.selectedPort))) {
            this.setBoardsConfig({
                selectedBoard: this.boardsConfig.selectedBoard,
                selectedPort: undefined,
            });
            this.onBoardsConfigChangedEmitter.fire(this._boardsConfig);
        }
        const boardsConfig = this.boardsConfig;
        const currentAvailableBoards = this._availableBoards;
        const availableBoards = [];
        const attachedBoards = this._attachedBoards.filter(({ port }) => !!port);
        const availableBoardPorts = availablePorts.filter(protocol_1.Port.visiblePorts(attachedBoards));
        for (const boardPort of availableBoardPorts) {
            const board = attachedBoards.find(({ port }) => protocol_1.Port.sameAs(boardPort, port));
            // "board" will always be falsey for
            // port that was originally mapped
            // to unknown board and then selected
            // manually by user
            const lastSelectedBoard = await this.getLastSelectedBoardOnPort(boardPort);
            let availableBoard = {};
            if (board) {
                availableBoard = Object.assign(Object.assign({}, board), { state: AvailableBoard.State.recognized, selected: boards_config_1.BoardsConfig.Config.sameAs(boardsConfig, board), port: boardPort });
            }
            else if (lastSelectedBoard) {
                // If the selected board is not recognized because it is a 3rd party board: https://github.com/arduino/arduino-cli/issues/623
                // We still want to show it without the red X in the boards toolbar: https://github.com/arduino/arduino-pro-ide/issues/198#issuecomment-599355836
                availableBoard = Object.assign(Object.assign({}, lastSelectedBoard), { state: AvailableBoard.State.guessed, selected: boards_config_1.BoardsConfig.Config.sameAs(boardsConfig, lastSelectedBoard) &&
                        protocol_1.Port.sameAs(boardPort, boardsConfig.selectedPort), port: boardPort });
            }
            else {
                availableBoard = {
                    name: nls_1.Unknown,
                    port: boardPort,
                    state: AvailableBoard.State.incomplete,
                };
            }
            availableBoards.push(availableBoard);
        }
        if (boardsConfig.selectedBoard &&
            availableBoards.every(({ selected }) => !selected)) {
            // If the selected board has the same port of an unknown board
            // that is already in availableBoards we might get a duplicate port.
            // So we remove the one already in the array and add the selected one.
            const found = availableBoards.findIndex((board) => { var _a, _b; return ((_a = board.port) === null || _a === void 0 ? void 0 : _a.address) === ((_b = boardsConfig.selectedPort) === null || _b === void 0 ? void 0 : _b.address); });
            if (found >= 0) {
                availableBoards.splice(found, 1);
            }
            availableBoards.push(Object.assign(Object.assign({}, boardsConfig.selectedBoard), { port: boardsConfig.selectedPort, selected: true, state: AvailableBoard.State.incomplete }));
        }
        availableBoards.sort(AvailableBoard.compare);
        let hasChanged = availableBoards.length !== currentAvailableBoards.length;
        for (let i = 0; !hasChanged && i < availableBoards.length; i++) {
            const [left, right] = [availableBoards[i], currentAvailableBoards[i]];
            hasChanged =
                left.fqbn !== right.fqbn ||
                    !!AvailableBoard.compare(left, right) ||
                    left.selected !== right.selected;
        }
        if (hasChanged) {
            this._availableBoards = availableBoards;
            this.onAvailableBoardsChangedEmitter.fire(this._availableBoards);
        }
    }
    async getLastSelectedBoardOnPort(port) {
        const key = this.getLastSelectedBoardOnPortKey(port);
        return this.getData(key);
    }
    async saveState() {
        // We save the port with the selected board name/FQBN, to be able to guess a better board name.
        // Required when the attached board belongs to a 3rd party boards package, and neither the name, nor
        // the FQBN can be retrieved with a `board list` command.
        // https://github.com/arduino/arduino-cli/issues/623
        const { selectedBoard, selectedPort } = this.boardsConfig;
        if (selectedBoard && selectedPort) {
            const key = this.getLastSelectedBoardOnPortKey(selectedPort);
            await this.setData(key, selectedBoard);
        }
        await Promise.all([
            this.setData('latest-valid-boards-config', this.latestValidBoardsConfig),
            this.setData('latest-boards-config', this.latestBoardsConfig),
        ]);
    }
    getLastSelectedBoardOnPortKey(port) {
        // TODO: we lose the port's `protocol` info (`serial`, `network`, etc.) here if the `port` is a `string`.
        return `last-selected-board-on-port:${typeof port === 'string' ? port : port.address}`;
    }
    async loadState() {
        const storedLatestValidBoardsConfig = await this.getData('latest-valid-boards-config');
        if (storedLatestValidBoardsConfig) {
            this.latestValidBoardsConfig = storedLatestValidBoardsConfig;
            if (this.canUploadTo(this.latestValidBoardsConfig)) {
                this.boardsConfig = this.latestValidBoardsConfig;
            }
        }
        else {
            // If we could not restore the latest valid config, try to restore something, the board at least.
            let storedLatestBoardsConfig = await this.getData('latest-boards-config');
            // Try to get from the URL if it was not persisted.
            if (!storedLatestBoardsConfig) {
                storedLatestBoardsConfig = boards_config_1.BoardsConfig.Config.getConfig(new URL(window.location.href));
            }
            if (storedLatestBoardsConfig) {
                this.latestBoardsConfig = storedLatestBoardsConfig;
                this.boardsConfig = this.latestBoardsConfig;
            }
        }
    }
    setData(key, value) {
        return this.commandService.executeCommand(storage_wrapper_1.StorageWrapper.Commands.SET_DATA.id, key, value);
    }
    getData(key) {
        return this.commandService.executeCommand(storage_wrapper_1.StorageWrapper.Commands.GET_DATA.id, key);
    }
};
__decorate([
    (0, inversify_1.inject)(logger_1.ILogger),
    __metadata("design:type", Object)
], BoardsServiceProvider.prototype, "logger", void 0);
__decorate([
    (0, inversify_1.inject)(message_service_1.MessageService),
    __metadata("design:type", message_service_1.MessageService)
], BoardsServiceProvider.prototype, "messageService", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.BoardsService),
    __metadata("design:type", Object)
], BoardsServiceProvider.prototype, "boardsService", void 0);
__decorate([
    (0, inversify_1.inject)(command_1.CommandService),
    __metadata("design:type", Object)
], BoardsServiceProvider.prototype, "commandService", void 0);
__decorate([
    (0, inversify_1.inject)(notification_center_1.NotificationCenter),
    __metadata("design:type", notification_center_1.NotificationCenter)
], BoardsServiceProvider.prototype, "notificationCenter", void 0);
__decorate([
    (0, inversify_1.inject)(frontend_application_state_1.FrontendApplicationStateService),
    __metadata("design:type", frontend_application_state_1.FrontendApplicationStateService)
], BoardsServiceProvider.prototype, "appStateService", void 0);
BoardsServiceProvider = __decorate([
    (0, inversify_1.injectable)()
], BoardsServiceProvider);
exports.BoardsServiceProvider = BoardsServiceProvider;
var AvailableBoard;
(function (AvailableBoard) {
    let State;
    (function (State) {
        /**
         * Retrieved from the CLI via the `board list` command.
         */
        State[State["recognized"] = 0] = "recognized";
        /**
         * Guessed the name/FQBN of the board from the available board ports (3rd party).
         */
        State[State["guessed"] = 1] = "guessed";
        /**
         * We do not know anything about this board, probably a 3rd party. The user has not selected a board for this port yet.
         */
        State[State["incomplete"] = 2] = "incomplete";
    })(State = AvailableBoard.State || (AvailableBoard.State = {}));
    function is(board) {
        return protocol_1.Board.is(board) && 'state' in board;
    }
    AvailableBoard.is = is;
    function hasPort(board) {
        return !!board.port;
    }
    AvailableBoard.hasPort = hasPort;
    // Available boards must be sorted in this order:
    // 1. Serial with recognized boards
    // 2. Serial with guessed boards
    // 3. Serial with incomplete boards
    // 4. Network with recognized boards
    // 5. Other protocols with recognized boards
    AvailableBoard.compare = (left, right) => {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l, _m;
        if (((_a = left.port) === null || _a === void 0 ? void 0 : _a.protocol) === 'serial' && ((_b = right.port) === null || _b === void 0 ? void 0 : _b.protocol) !== 'serial') {
            return -1;
        }
        else if (((_c = left.port) === null || _c === void 0 ? void 0 : _c.protocol) !== 'serial' &&
            ((_d = right.port) === null || _d === void 0 ? void 0 : _d.protocol) === 'serial') {
            return 1;
        }
        else if (((_e = left.port) === null || _e === void 0 ? void 0 : _e.protocol) === 'network' &&
            ((_f = right.port) === null || _f === void 0 ? void 0 : _f.protocol) !== 'network') {
            return -1;
        }
        else if (((_g = left.port) === null || _g === void 0 ? void 0 : _g.protocol) !== 'network' &&
            ((_h = right.port) === null || _h === void 0 ? void 0 : _h.protocol) === 'network') {
            return 1;
        }
        else if (((_j = left.port) === null || _j === void 0 ? void 0 : _j.protocol) === ((_k = right.port) === null || _k === void 0 ? void 0 : _k.protocol)) {
            // We show all ports, including those that have guessed
            // or unrecognized boards, so we must sort those too.
            if (left.state < right.state) {
                return -1;
            }
            else if (left.state > right.state) {
                return 1;
            }
        }
        return (0, utils_1.naturalCompare)((_l = left.port) === null || _l === void 0 ? void 0 : _l.address, (_m = right.port) === null || _m === void 0 ? void 0 : _m.address);
    };
})(AvailableBoard = exports.AvailableBoard || (exports.AvailableBoard = {}));
//# sourceMappingURL=boards-service-provider.js.map