"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardsToolBarItem = exports.BoardsDropDown = void 0;
const React = require("@theia/core/shared/react");
const ReactDOM = require("@theia/core/shared/react-dom");
const disposable_1 = require("@theia/core/lib/common/disposable");
const open_boards_config_1 = require("../contributions/open-boards-config");
const boards_service_provider_1 = require("./boards-service-provider");
const common_1 = require("@theia/core/lib/common");
const classnames_1 = require("classnames");
class BoardsDropDown extends React.Component {
    constructor(props) {
        super(props);
        this.listRef = React.createRef();
        let list = document.getElementById('boards-dropdown-container');
        if (!list) {
            list = document.createElement('div');
            list.id = 'boards-dropdown-container';
            document.body.appendChild(list);
            this.dropdownElement = list;
        }
    }
    componentDidUpdate(prevProps) {
        if (prevProps.coords === 'hidden' && this.listRef.current) {
            this.listRef.current.focus();
        }
    }
    render() {
        return ReactDOM.createPortal(this.renderNode(), this.dropdownElement);
    }
    renderNode() {
        const { coords, items } = this.props;
        if (coords === 'hidden') {
            return '';
        }
        const footerLabel = common_1.nls.localize('arduino/board/openBoardsConfig', 'Select other board and port…');
        return (React.createElement("div", { className: "arduino-boards-dropdown-list", style: Object.assign({ position: 'absolute' }, coords), ref: this.listRef, tabIndex: 0 },
            React.createElement("div", { className: "arduino-boards-dropdown-list--items-container" }, items
                .map(({ name, port, selected, onClick }) => ({
                boardLabel: name,
                port,
                selected,
                onClick,
            }))
                .map(this.renderItem)),
            React.createElement("div", { key: footerLabel, tabIndex: 0, className: "arduino-boards-dropdown-item arduino-board-dropdown-footer", onClick: () => this.props.openBoardsConfig() },
                React.createElement("div", null, footerLabel))));
    }
    renderItem({ boardLabel, port, selected, onClick, }) {
        const protocolIcon = iconNameFromProtocol(port.protocol);
        const onKeyUp = (e) => {
            if (e.key === 'Enter') {
                onClick();
            }
        };
        return (React.createElement("div", { key: `board-item--${boardLabel}-${port.address}`, className: (0, classnames_1.default)('arduino-boards-dropdown-item', {
                'arduino-boards-dropdown-item--selected': selected,
            }), onClick: onClick, onKeyUp: onKeyUp, tabIndex: 0 },
            React.createElement("div", { className: (0, classnames_1.default)('arduino-boards-dropdown-item--protocol', 'fa', protocolIcon) }),
            React.createElement("div", { className: "arduino-boards-dropdown-item--label", title: `${boardLabel}\n${port.address}` },
                React.createElement("div", { className: "arduino-boards-dropdown-item--board-label noWrapInfo noselect" }, boardLabel),
                React.createElement("div", { className: "arduino-boards-dropdown-item--port-label noWrapInfo noselect" }, port.addressLabel)),
            selected ? React.createElement("div", { className: "fa fa-check" }) : ''));
    }
}
exports.BoardsDropDown = BoardsDropDown;
class BoardsToolBarItem extends React.Component {
    constructor(props) {
        super(props);
        this.toDispose = new disposable_1.DisposableCollection();
        this.show = (event) => {
            const { currentTarget: element } = event;
            if (element instanceof HTMLElement) {
                if (this.state.coords === 'hidden') {
                    const rect = element.getBoundingClientRect();
                    this.setState({
                        coords: {
                            top: rect.top,
                            left: rect.left,
                            width: rect.width,
                            paddingTop: rect.height,
                        },
                    });
                }
                else {
                    this.setState({ coords: 'hidden' });
                }
            }
            event.stopPropagation();
            event.nativeEvent.stopImmediatePropagation();
        };
        this.openDialog = async (previousBoardConfig) => {
            const selectedBoardConfig = await this.props.commands.executeCommand(open_boards_config_1.OpenBoardsConfig.Commands.OPEN_DIALOG.id);
            if (previousBoardConfig &&
                (!(selectedBoardConfig === null || selectedBoardConfig === void 0 ? void 0 : selectedBoardConfig.selectedPort) ||
                    !(selectedBoardConfig === null || selectedBoardConfig === void 0 ? void 0 : selectedBoardConfig.selectedBoard))) {
                this.props.boardsServiceProvider.boardsConfig = previousBoardConfig;
            }
        };
        const { availableBoards } = props.boardsServiceProvider;
        this.state = {
            availableBoards,
            coords: 'hidden',
        };
        document.addEventListener('click', () => {
            this.setState({ coords: 'hidden' });
        });
    }
    componentDidMount() {
        this.props.boardsServiceProvider.onAvailableBoardsChanged((availableBoards) => this.setState({ availableBoards }));
    }
    componentWillUnmount() {
        this.toDispose.dispose();
    }
    render() {
        const { coords, availableBoards } = this.state;
        const { selectedBoard, selectedPort } = this.props.boardsServiceProvider.boardsConfig;
        const boardLabel = (selectedBoard === null || selectedBoard === void 0 ? void 0 : selectedBoard.name) ||
            common_1.nls.localize('arduino/board/selectBoard', 'Select Board');
        const selectedPortLabel = portLabel(selectedPort === null || selectedPort === void 0 ? void 0 : selectedPort.address);
        const isConnected = Boolean(selectedBoard && selectedPort);
        const protocolIcon = isConnected
            ? iconNameFromProtocol((selectedPort === null || selectedPort === void 0 ? void 0 : selectedPort.protocol) || '')
            : null;
        const protocolIconClassNames = (0, classnames_1.default)('arduino-boards-toolbar-item--protocol', 'fa', protocolIcon);
        return (React.createElement(React.Fragment, null,
            React.createElement("div", { className: "arduino-boards-toolbar-item-container", title: selectedPortLabel, onClick: this.show },
                protocolIcon && React.createElement("div", { className: protocolIconClassNames }),
                React.createElement("div", { className: (0, classnames_1.default)('arduino-boards-toolbar-item--label', 'noWrapInfo', 'noselect', { 'arduino-boards-toolbar-item--label-connected': isConnected }) }, boardLabel),
                React.createElement("div", { className: "fa fa-caret-down caret" })),
            React.createElement(BoardsDropDown, { coords: coords, items: availableBoards
                    .filter(boards_service_provider_1.AvailableBoard.hasPort)
                    .map((board) => (Object.assign(Object.assign({}, board), { onClick: () => {
                        if (!board.fqbn) {
                            const previousBoardConfig = this.props.boardsServiceProvider.boardsConfig;
                            this.props.boardsServiceProvider.boardsConfig = {
                                selectedPort: board.port,
                            };
                            this.openDialog(previousBoardConfig);
                        }
                        else {
                            this.props.boardsServiceProvider.boardsConfig = {
                                selectedBoard: board,
                                selectedPort: board.port,
                            };
                        }
                        this.setState({ coords: 'hidden' });
                    } }))), openBoardsConfig: this.openDialog })));
    }
}
exports.BoardsToolBarItem = BoardsToolBarItem;
function iconNameFromProtocol(protocol) {
    switch (protocol) {
        case 'serial':
            return 'fa-arduino-technology-usb';
        case 'network':
            return 'fa-arduino-technology-connection';
        /*
          Bluetooth ports are not listed yet from the CLI;
          Not sure about the naming ('bluetooth'); make sure it's correct before uncommenting the following lines
        */
        // case 'bluetooth':
        //   return 'fa-arduino-technology-bluetooth';
        default:
            return 'fa-arduino-technology-3dimensionscube';
    }
}
function portLabel(portName) {
    return portName
        ? common_1.nls.localize('arduino/board/portLabel', 'Port: {0}', portName)
        : common_1.nls.localize('arduino/board/disconnected', 'Disconnected');
}
//# sourceMappingURL=boards-toolbar-item.js.map