"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var Close_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Close = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const algorithm_1 = require("@theia/core/shared/@phosphor/algorithm");
const remote = require("@theia/core/electron-shared/@electron/remote");
const monaco_editor_1 = require("@theia/monaco/lib/browser/monaco-editor");
const nls_1 = require("@theia/core/lib/common/nls");
const arduino_menus_1 = require("../menu/arduino-menus");
const contribution_1 = require("./contribution");
const dialogs_1 = require("@theia/core/lib/browser/dialogs");
const sketches_service_client_impl_1 = require("../../common/protocol/sketches-service-client-impl");
const save_as_sketch_1 = require("./save-as-sketch");
/**
 * Closes the `current` closeable editor, or any closeable current widget from the main area, or the current sketch window.
 */
let Close = Close_1 = class Close extends contribution_1.SketchContribution {
    onStart(app) {
        this.shell = app.shell;
    }
    registerCommands(registry) {
        registry.registerCommand(Close_1.Commands.CLOSE, {
            execute: () => {
                // Close current editor if closeable.
                const { currentEditor } = this.editorManager;
                if (currentEditor && currentEditor.title.closable) {
                    currentEditor.close();
                    return;
                }
                if (this.shell) {
                    // Close current widget from the main area if possible.
                    const { currentWidget } = this.shell;
                    if (currentWidget) {
                        const currentWidgetInMain = (0, algorithm_1.toArray)(this.shell.mainPanel.widgets()).find((widget) => widget === currentWidget);
                        if (currentWidgetInMain && currentWidgetInMain.title.closable) {
                            return currentWidgetInMain.close();
                        }
                    }
                }
                return remote.getCurrentWindow().close();
            },
        });
    }
    registerMenus(registry) {
        registry.registerMenuAction(arduino_menus_1.ArduinoMenus.FILE__SKETCH_GROUP, {
            commandId: Close_1.Commands.CLOSE.id,
            label: nls_1.nls.localize('vscode/editor.contribution/close', 'Close'),
            order: '5',
        });
    }
    registerKeybindings(registry) {
        registry.registerKeybinding({
            command: Close_1.Commands.CLOSE.id,
            keybinding: 'CtrlCmd+W',
        });
    }
    // `FrontendApplicationContribution#onWillStop`
    onWillStop() {
        return {
            reason: 'save-sketch',
            action: () => {
                return this.showSaveSketchDialog();
            },
        };
    }
    /**
     * If returns with `true`, IDE2 will close. Otherwise, it won't.
     */
    async showSaveSketchDialog() {
        const sketch = await this.isCurrentSketchTemp();
        if (!sketch) {
            // Normal close workflow: if there are dirty editors prompt the user.
            if (!this.shell) {
                console.error(`Could not get the application shell. Something went wrong.`);
                return true;
            }
            if (this.shell.canSaveAll()) {
                const prompt = await this.prompt(false);
                switch (prompt) {
                    case Prompt.DoNotSave:
                        return true;
                    case Prompt.Cancel:
                        return false;
                    case Prompt.Save: {
                        await this.shell.saveAll();
                        return true;
                    }
                    default:
                        throw new Error(`Unexpected prompt: ${prompt}`);
                }
            }
            return true;
        }
        // If non of the sketch files were ever touched, do not prompt the save dialog. (#1274)
        const wereTouched = await Promise.all(contribution_1.Sketch.uris(sketch).map((uri) => this.wasTouched(uri)));
        if (wereTouched.every((wasTouched) => !Boolean(wasTouched))) {
            return true;
        }
        const prompt = await this.prompt(true);
        switch (prompt) {
            case Prompt.DoNotSave:
                return true;
            case Prompt.Cancel:
                return false;
            case Prompt.Save: {
                // If `save as` was canceled by user, the result will be `undefined`, otherwise the new URI.
                const result = await this.commandService.executeCommand(save_as_sketch_1.SaveAsSketch.Commands.SAVE_AS_SKETCH.id, {
                    execOnlyIfTemp: false,
                    openAfterMove: false,
                    wipeOriginal: true,
                    markAsRecentlyOpened: true,
                });
                return !!result;
            }
            default:
                throw new Error(`Unexpected prompt: ${prompt}`);
        }
    }
    async prompt(isTemp) {
        const { response } = await remote.dialog.showMessageBox(remote.getCurrentWindow(), {
            message: nls_1.nls.localize('arduino/sketch/saveSketch', 'Save your sketch to open it again later.'),
            title: nls_1.nls.localize('theia/core/quitTitle', 'Are you sure you want to quit?'),
            type: 'question',
            buttons: [
                nls_1.nls.localizeByDefault("Don't Save"),
                dialogs_1.Dialog.CANCEL,
                nls_1.nls.localizeByDefault(isTemp ? 'Save As...' : 'Save'),
            ],
            defaultId: 2, // `Save`/`Save As...` button index is the default.
        });
        switch (response) {
            case 0:
                return Prompt.DoNotSave;
            case 1:
                return Prompt.Cancel;
            case 2:
                return Prompt.Save;
            default:
                throw new Error(`Unexpected response: ${response}`);
        }
    }
    async isCurrentSketchTemp() {
        const currentSketch = await this.sketchServiceClient.currentSketch();
        if (sketches_service_client_impl_1.CurrentSketch.isValid(currentSketch)) {
            const isTemp = await this.sketchService.isTemp(currentSketch);
            if (isTemp) {
                return currentSketch;
            }
        }
        return false;
    }
    /**
     * If the file was ever touched/modified. We get this based on the `version` of the monaco model.
     */
    async wasTouched(uri) {
        var _a;
        const editorWidget = await this.editorManager.getByUri(new contribution_1.URI(uri));
        if (editorWidget) {
            const { editor } = editorWidget;
            if (editor instanceof monaco_editor_1.MonacoEditor) {
                const versionId = (_a = editor.getControl().getModel()) === null || _a === void 0 ? void 0 : _a.getVersionId();
                if (this.isInteger(versionId) && versionId > 1) {
                    return true;
                }
            }
        }
        return false;
    }
    isInteger(arg) {
        return Number.isInteger(arg);
    }
};
Close = Close_1 = __decorate([
    (0, inversify_1.injectable)()
], Close);
exports.Close = Close;
var Prompt;
(function (Prompt) {
    Prompt[Prompt["Save"] = 0] = "Save";
    Prompt[Prompt["DoNotSave"] = 1] = "DoNotSave";
    Prompt[Prompt["Cancel"] = 2] = "Cancel";
})(Prompt || (Prompt = {}));
(function (Close) {
    let Commands;
    (function (Commands) {
        Commands.CLOSE = {
            id: 'arduino-close',
        };
    })(Commands = Close.Commands || (Close.Commands = {}));
})(Close = exports.Close || (exports.Close = {}));
exports.Close = Close;
//# sourceMappingURL=close.js.map