"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FilterableListContainer = void 0;
const React = require("@theia/core/shared/react");
const debounce = require("lodash.debounce");
const dialogs_1 = require("@theia/core/lib/browser/dialogs");
const progressible_1 = require("../../../common/protocol/progressible");
const search_bar_1 = require("./search-bar");
const component_list_1 = require("./component-list");
const common_1 = require("@theia/core/lib/common");
class FilterableListContainer extends React.Component {
    constructor(props) {
        super(props);
        this.handlePropChange = (prop, value) => {
            const searchOptions = Object.assign(Object.assign({}, this.state.searchOptions), { [prop]: value });
            this.setSearchOptionsAndUpdate(searchOptions);
        };
        this.state = {
            searchOptions: props.defaultSearchOptions,
            items: [],
        };
    }
    componentDidMount() {
        this.search = debounce(this.search, 500);
        this.search(this.state.searchOptions);
        this.props.searchOptionsDidChange((newSearchOptions) => {
            const { searchOptions } = this.state;
            this.setSearchOptionsAndUpdate(Object.assign(Object.assign({}, searchOptions), newSearchOptions));
        });
    }
    componentDidUpdate() {
        // See: arduino/arduino-pro-ide#101
        // Resets the top of the perfect scroll-bar's thumb.
        this.props.container.updateScrollBar();
    }
    render() {
        return (React.createElement("div", { className: 'filterable-list-container' },
            this.renderSearchBar(),
            this.renderSearchFilter(),
            React.createElement("div", { className: "filterable-list-container" }, this.renderComponentList())));
    }
    renderSearchFilter() {
        return (React.createElement(React.Fragment, null, this.props.filterRenderer.render(this.state.searchOptions, this.handlePropChange.bind(this))));
    }
    renderSearchBar() {
        var _a;
        return (React.createElement(search_bar_1.SearchBar, { resolveFocus: this.props.resolveFocus, filterText: (_a = this.state.searchOptions.query) !== null && _a !== void 0 ? _a : '', onFilterTextChanged: (query) => this.handlePropChange('query', query) }));
    }
    renderComponentList() {
        const { itemLabel, itemDeprecated, itemRenderer } = this.props;
        return (React.createElement(component_list_1.ComponentList, { items: this.state.items, itemLabel: itemLabel, itemDeprecated: itemDeprecated, itemRenderer: itemRenderer, install: this.install.bind(this), uninstall: this.uninstall.bind(this) }));
    }
    setSearchOptionsAndUpdate(searchOptions) {
        this.setState({ searchOptions }, () => this.search(searchOptions));
    }
    search(searchOptions) {
        const { searchable } = this.props;
        searchable
            .search(searchOptions)
            .then((items) => this.setState({ items: this.sort(items) }));
    }
    sort(items) {
        const { itemLabel, itemDeprecated } = this.props;
        return items.sort((left, right) => {
            // always put deprecated items at the bottom of the list
            if (itemDeprecated(left)) {
                return 1;
            }
            return itemLabel(left).localeCompare(itemLabel(right));
        });
    }
    async install(item, version) {
        const { install, searchable } = this.props;
        await progressible_1.ExecuteWithProgress.doWithProgress(Object.assign(Object.assign({}, this.props), { progressText: common_1.nls.localize('arduino/common/processing', 'Processing') +
                ` ${item.name}:${version}`, run: ({ progressId }) => install({ item, progressId, version }) }));
        const items = await searchable.search(this.state.searchOptions);
        this.setState({ items: this.sort(items) });
    }
    async uninstall(item) {
        const ok = await new dialogs_1.ConfirmDialog({
            title: common_1.nls.localize('arduino/component/uninstall', 'Uninstall'),
            msg: common_1.nls.localize('arduino/component/uninstallMsg', 'Do you want to uninstall {0}?', item.name),
            ok: common_1.nls.localize('vscode/extensionsUtils/yes', 'Yes'),
            cancel: common_1.nls.localize('vscode/extensionsUtils/no', 'No'),
        }).open();
        if (!ok) {
            return;
        }
        const { uninstall, searchable } = this.props;
        await progressible_1.ExecuteWithProgress.doWithProgress(Object.assign(Object.assign({}, this.props), { progressText: common_1.nls.localize('arduino/common/processing', 'Processing') +
                ` ${item.name}${item.installedVersion ? `:${item.installedVersion}` : ''}`, run: ({ progressId }) => uninstall({ item, progressId }) }));
        const items = await searchable.search(this.state.searchOptions);
        this.setState({ items: this.sort(items) });
    }
}
exports.FilterableListContainer = FilterableListContainer;
//# sourceMappingURL=filterable-list-container.js.map