"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Board = exports.Programmer = exports.ConfigOption = exports.InstalledBoardWithPackage = exports.BoardWithPackage = exports.BoardsPackage = exports.Port = exports.BoardSearch = exports.BoardsService = exports.BoardsServicePath = exports.AttachedBoardsChangeEvent = exports.AvailablePorts = void 0;
const utils_1 = require("./../utils");
const nls_1 = require("@theia/core/lib/common/nls");
const nls_2 = require("../nls");
var AvailablePorts;
(function (AvailablePorts) {
    function groupByProtocol(availablePorts) {
        const grouped = new Map();
        for (const portID of Object.keys(availablePorts)) {
            const [port, boards] = availablePorts[portID];
            let ports = grouped.get(port.protocol);
            if (!ports) {
                ports = {};
            }
            ports[portID] = [port, boards];
            grouped.set(port.protocol, ports);
        }
        return grouped;
    }
    AvailablePorts.groupByProtocol = groupByProtocol;
    function split(state) {
        const availablePorts = [];
        const attachedBoards = [];
        for (const key of Object.keys(state)) {
            const [port, boards] = state[key];
            availablePorts.push(port);
            attachedBoards.push(...boards);
        }
        return {
            boards: attachedBoards,
            ports: availablePorts,
        };
    }
    AvailablePorts.split = split;
})(AvailablePorts = exports.AvailablePorts || (exports.AvailablePorts = {}));
var AttachedBoardsChangeEvent;
(function (AttachedBoardsChangeEvent) {
    function isEmpty(event) {
        const { detached, attached } = diff(event);
        return (!!detached.boards.length &&
            !!detached.ports.length &&
            !!attached.boards.length &&
            !!attached.ports.length);
    }
    AttachedBoardsChangeEvent.isEmpty = isEmpty;
    function toString(event) {
        const rows = [];
        if (!isEmpty(event)) {
            const { attached, detached } = diff(event);
            const visitedAttachedPorts = [];
            const visitedDetachedPorts = [];
            for (const board of attached.boards) {
                const port = board.port ? ` on ${Port.toString(board.port)}` : '';
                rows.push(` - Attached board: ${Board.toString(board)}${port}`);
                if (board.port) {
                    visitedAttachedPorts.push(board.port);
                }
            }
            for (const board of detached.boards) {
                const port = board.port ? ` from ${Port.toString(board.port)}` : '';
                rows.push(` - Detached board: ${Board.toString(board)}${port}`);
                if (board.port) {
                    visitedDetachedPorts.push(board.port);
                }
            }
            for (const port of attached.ports) {
                if (!visitedAttachedPorts.find((p) => Port.sameAs(port, p))) {
                    rows.push(` - New port is available on ${Port.toString(port)}`);
                }
            }
            for (const port of detached.ports) {
                if (!visitedDetachedPorts.find((p) => Port.sameAs(port, p))) {
                    rows.push(` - Port is no longer available on ${Port.toString(port)}`);
                }
            }
        }
        return rows.length ? rows.join('\n') : 'No changes.';
    }
    AttachedBoardsChangeEvent.toString = toString;
    function diff(event) {
        // In `lefts` AND not in `rights`.
        const diff = (lefts, rights, sameAs) => {
            return lefts.filter((left) => rights.findIndex((right) => sameAs(left, right)) === -1);
        };
        const { boards: newBoards } = event.newState;
        const { boards: oldBoards } = event.oldState;
        const { ports: newPorts } = event.newState;
        const { ports: oldPorts } = event.oldState;
        const boardSameAs = (left, right) => Board.sameAs(left, right);
        const portSameAs = (left, right) => Port.sameAs(left, right);
        return {
            detached: {
                boards: diff(oldBoards, newBoards, boardSameAs),
                ports: diff(oldPorts, newPorts, portSameAs),
            },
            attached: {
                boards: diff(newBoards, oldBoards, boardSameAs),
                ports: diff(newPorts, oldPorts, portSameAs),
            },
        };
    }
    AttachedBoardsChangeEvent.diff = diff;
})(AttachedBoardsChangeEvent = exports.AttachedBoardsChangeEvent || (exports.AttachedBoardsChangeEvent = {}));
exports.BoardsServicePath = '/services/boards-service';
exports.BoardsService = Symbol('BoardsService');
var BoardSearch;
(function (BoardSearch) {
    BoardSearch.TypeLiterals = [
        'All',
        'Updatable',
        'Arduino',
        'Contributed',
        'Arduino Certified',
        'Partner',
        'Arduino@Heart',
    ];
    BoardSearch.TypeLabels = {
        All: nls_2.All,
        Updatable: nls_2.Updatable,
        Arduino: 'Arduino',
        Contributed: nls_2.Contributed,
        'Arduino Certified': nls_1.nls.localize('arduino/boardsType/arduinoCertified', 'Arduino Certified'),
        Partner: nls_2.Partner,
        'Arduino@Heart': 'Arduino@Heart',
    };
    BoardSearch.PropertyLabels = {
        type: nls_2.Type,
    };
})(BoardSearch = exports.BoardSearch || (exports.BoardSearch = {}));
var Port;
(function (Port) {
    let Properties;
    (function (Properties) {
        function create(properties) {
            if (!properties) {
                return {};
            }
            return properties.reduce((acc, curr) => {
                const [key, value] = curr;
                acc[key] = value;
                return acc;
            }, {});
        }
        Properties.create = create;
    })(Properties = Port.Properties || (Port.Properties = {}));
    function is(arg) {
        if (typeof arg === 'object') {
            // eslint-disable-next-line @typescript-eslint/no-explicit-any
            const object = arg;
            return ('address' in object &&
                typeof object['address'] === 'string' &&
                'addressLabel' in object &&
                typeof object['addressLabel'] === 'string' &&
                'protocol' in object &&
                typeof object['protocol'] === 'string' &&
                'protocolLabel' in object &&
                typeof object['protocolLabel'] === 'string');
        }
        return false;
    }
    Port.is = is;
    /**
     * Key is the combination of address and protocol formatted like `'${address}|${protocol}'` used to uniquely identify a port.
     */
    function keyOf({ address, protocol }) {
        return `${address}|${protocol}`;
    }
    Port.keyOf = keyOf;
    function toString({ addressLabel, protocolLabel }) {
        return `${addressLabel} ${protocolLabel}`;
    }
    Port.toString = toString;
    function compare(left, right) {
        // Ports must be sorted in this order:
        // 1. Serial
        // 2. Network
        // 3. Other protocols
        if (left.protocol === 'serial' && right.protocol !== 'serial') {
            return -1;
        }
        else if (left.protocol !== 'serial' && right.protocol === 'serial') {
            return 1;
        }
        else if (left.protocol === 'network' && right.protocol !== 'network') {
            return -1;
        }
        else if (left.protocol !== 'network' && right.protocol === 'network') {
            return 1;
        }
        return (0, utils_1.naturalCompare)(left.address, right.address);
    }
    Port.compare = compare;
    function sameAs(left, right) {
        if (left && right) {
            return left.address === right.address && left.protocol === right.protocol;
        }
        return false;
    }
    Port.sameAs = sameAs;
    // See https://github.com/arduino/arduino-ide/commit/79ea0fa9a6ad2b01eaac22cef2f494d3b68284e6#diff-fb37f20bea00881acee3aafddb1ecefcecf41ce59845ca1510da79e918ee0837L338-L348
    // See https://github.com/arduino/arduino-ide/commit/79ea0fa9a6ad2b01eaac22cef2f494d3b68284e6#diff-e42c82bb67e277cfa4598239952afd65db44dba55dc7d68df619dfccfa648279L441-L455
    // See https://github.com/arduino/arduino-ide/commit/74bfdc4c56d7a1577a4e800a378c21b82c1da5f8#diff-e42c82bb67e277cfa4598239952afd65db44dba55dc7d68df619dfccfa648279L405-R424
    /**
     * All ports with `'serial'` or `'network'` `protocol`, or any other port `protocol` that has at least one recognized board connected to.
     */
    function visiblePorts(boardsHaystack) {
        return (port) => {
            var _a;
            if (port.protocol === 'serial' || port.protocol === 'network') {
                // Allow all `serial` and `network` boards.
                // IDE2 must support better label for unrecognized `network` boards: https://github.com/arduino/arduino-ide/issues/1331
                return true;
            }
            // All other ports with different protocol are
            // only shown if there is a recognized board
            // connected
            for (const board of boardsHaystack) {
                if (((_a = board.port) === null || _a === void 0 ? void 0 : _a.address) === port.address) {
                    return true;
                }
            }
            return false;
        };
    }
    Port.visiblePorts = visiblePorts;
})(Port = exports.Port || (exports.Port = {}));
var BoardsPackage;
(function (BoardsPackage) {
    function equals(left, right) {
        return left.id === right.id;
    }
    BoardsPackage.equals = equals;
    function contains(selectedBoard, { id, boards }) {
        if (boards.some((board) => Board.sameAs(board, selectedBoard))) {
            return true;
        }
        if (selectedBoard.fqbn) {
            const [platform, architecture] = selectedBoard.fqbn.split(':');
            if (platform && architecture) {
                return `${platform}:${architecture}` === id;
            }
        }
        return false;
    }
    BoardsPackage.contains = contains;
})(BoardsPackage = exports.BoardsPackage || (exports.BoardsPackage = {}));
var BoardWithPackage;
(function (BoardWithPackage) {
    function is(board) {
        return !!board.packageId && !!board.packageName;
    }
    BoardWithPackage.is = is;
})(BoardWithPackage = exports.BoardWithPackage || (exports.BoardWithPackage = {}));
var InstalledBoardWithPackage;
(function (InstalledBoardWithPackage) {
    function is(boardWithPackage) {
        return !!boardWithPackage.fqbn;
    }
    InstalledBoardWithPackage.is = is;
})(InstalledBoardWithPackage = exports.InstalledBoardWithPackage || (exports.InstalledBoardWithPackage = {}));
var ConfigOption;
(function (ConfigOption) {
    function is(arg) {
        return (!!arg &&
            'option' in arg &&
            'label' in arg &&
            'values' in arg &&
            typeof arg['option'] === 'string' &&
            typeof arg['label'] === 'string' &&
            Array.isArray(arg['values']));
    }
    ConfigOption.is = is;
    /**
     * Appends the configuration options to the `fqbn` argument.
     * Throws an error if the `fqbn` does not have the `segment(':'segment)*` format.
     * The provided output format is always segment(':'segment)*(':'option'='value(','option'='value)*)?
     */
    function decorate(fqbn, configOptions) {
        if (!configOptions.length) {
            return fqbn;
        }
        const toValue = (values) => {
            const selectedValue = values.find(({ selected }) => selected);
            if (!selectedValue) {
                console.warn(`None of the config values was selected. Values were: ${JSON.stringify(values)}`);
                return undefined;
            }
            return selectedValue.value;
        };
        const options = configOptions
            .map(({ option, values }) => [option, toValue(values)])
            .filter(([, value]) => !!value)
            .map(([option, value]) => `${option}=${value}`)
            .join(',');
        return `${fqbn}:${options}`;
    }
    ConfigOption.decorate = decorate;
    class ConfigOptionError extends Error {
        constructor(message) {
            super(message);
            Object.setPrototypeOf(this, ConfigOptionError.prototype);
        }
    }
    ConfigOption.ConfigOptionError = ConfigOptionError;
    ConfigOption.LABEL_COMPARATOR = (left, right) => (0, utils_1.naturalCompare)(left.label.toLocaleLowerCase(), right.label.toLocaleLowerCase());
})(ConfigOption = exports.ConfigOption || (exports.ConfigOption = {}));
var Programmer;
(function (Programmer) {
    function equals(left, right) {
        if (!left) {
            return !right;
        }
        if (!right) {
            return !left;
        }
        return (left.id === right.id &&
            left.name === right.name &&
            left.platform === right.platform);
    }
    Programmer.equals = equals;
})(Programmer = exports.Programmer || (exports.Programmer = {}));
var Board;
(function (Board) {
    function is(board) {
        return !!board && 'name' in board;
    }
    Board.is = is;
    function equals(left, right) {
        return left.name === right.name && left.fqbn === right.fqbn;
    }
    Board.equals = equals;
    function sameAs(left, right) {
        // How to associate a selected board with one of the available cores: https://typefox.slack.com/archives/CJJHJCJSJ/p1571142327059200
        // 1. How to use the FQBN if any and infer the package ID from it: https://typefox.slack.com/archives/CJJHJCJSJ/p1571147549069100
        // 2. How to trim the `/Genuino` from the name: https://arduino.slack.com/archives/CJJHJCJSJ/p1571146951066800?thread_ts=1571142327.059200&cid=CJJHJCJSJ
        const other = typeof right === 'string' ? { name: right } : right;
        if (left.fqbn && other.fqbn) {
            return left.fqbn === other.fqbn;
        }
        return (left.name.replace('/Genuino', '') === other.name.replace('/Genuino', ''));
    }
    Board.sameAs = sameAs;
    function compare(left, right) {
        let result = (0, utils_1.naturalCompare)(left.name, right.name);
        if (result === 0) {
            result = (0, utils_1.naturalCompare)(left.fqbn || '', right.fqbn || '');
        }
        return result;
    }
    Board.compare = compare;
    function installed(board) {
        return !!board.fqbn;
    }
    Board.installed = installed;
    function toString(board, options = { useFqbn: true }) {
        const fqbn = options && options.useFqbn && board.fqbn ? ` [${board.fqbn}]` : '';
        return `${board.name}${fqbn}`;
    }
    Board.toString = toString;
    function decorateBoards(selectedBoard, boards) {
        // Board names are not unique. We show the corresponding core name as a detail.
        // https://github.com/arduino/arduino-cli/pull/294#issuecomment-513764948
        const distinctBoardNames = new Map();
        for (const { name } of boards) {
            const counter = distinctBoardNames.get(name) || 0;
            distinctBoardNames.set(name, counter + 1);
        }
        // Due to the non-unique board names, we have to check the package name as well.
        const selected = (board) => {
            if (!!selectedBoard) {
                if (Board.equals(board, selectedBoard)) {
                    if ('packageName' in selectedBoard) {
                        return board.packageName === selectedBoard.packageName;
                    }
                    if ('packageId' in selectedBoard) {
                        return board.packageId === selectedBoard.packageId;
                    }
                    return true;
                }
            }
            return false;
        };
        return boards.map((board) => (Object.assign(Object.assign({}, board), { details: (distinctBoardNames.get(board.name) || 0) > 1
                ? ` - ${board.packageName}`
                : undefined, selected: selected(board), missing: !installed(board) })));
    }
    Board.decorateBoards = decorateBoards;
})(Board = exports.Board || (exports.Board = {}));
//# sourceMappingURL=boards-service.js.map