"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Installable = void 0;
const semver = require("semver");
const progressible_1 = require("./progressible");
const utils_1 = require("../utils");
var Installable;
(function (Installable) {
    let Version;
    (function (Version) {
        /**
         * Most recent version comes first, then the previous versions. (`1.8.1`, `1.6.3`, `1.6.2`, `1.6.1` and so on.)
         *
         * If `coerce` is `true` tries to convert any invalid semver strings to a valid semver based on [these](https://github.com/npm/node-semver#coercion) rules.
         */
        Version.COMPARATOR = (left, right, coerce = false) => {
            const validLeft = semver.parse(left);
            const validRight = semver.parse(right);
            if (validLeft && validRight) {
                return semver.compare(validLeft, validRight);
            }
            if (coerce) {
                const coercedLeft = validLeft !== null && validLeft !== void 0 ? validLeft : semver.coerce(left);
                const coercedRight = validRight !== null && validRight !== void 0 ? validRight : semver.coerce(right);
                if (coercedLeft && coercedRight) {
                    return semver.compare(coercedLeft, coercedRight);
                }
            }
            return (0, utils_1.naturalCompare)(left, right);
        };
    })(Version = Installable.Version || (Installable.Version = {}));
    Installable.Installed = ({ installedVersion, }) => {
        return !!installedVersion;
    };
    Installable.Updateable = (item) => {
        const { installedVersion } = item;
        if (!installedVersion) {
            return false;
        }
        const latestVersion = item.availableVersions[0];
        if (!latestVersion) {
            console.warn(`Installed version ${installedVersion} is available for ${item.name}, but no available versions were available. Skipping.`);
            return false;
        }
        const result = Installable.Version.COMPARATOR(latestVersion, installedVersion, true);
        return result > 0;
    };
    async function installWithProgress(options) {
        const { item, version } = options;
        return progressible_1.ExecuteWithProgress.doWithProgress(Object.assign(Object.assign({}, options), { progressText: `Processing ${item.name}:${version}`, run: ({ progressId }) => options.installable.install({
                item: options.item,
                version: options.version,
                progressId,
            }) }));
    }
    Installable.installWithProgress = installWithProgress;
    async function uninstallWithProgress(options) {
        const { item } = options;
        return progressible_1.ExecuteWithProgress.doWithProgress(Object.assign(Object.assign({}, options), { progressText: `Processing ${item.name}${item.installedVersion ? `:${item.installedVersion}` : ''}`, run: ({ progressId }) => options.installable.uninstall({
                item: options.item,
                progressId,
            }) }));
    }
    Installable.uninstallWithProgress = uninstallWithProgress;
})(Installable = exports.Installable || (exports.Installable = {}));
//# sourceMappingURL=installable.js.map