"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SketchContainer = exports.Sketch = exports.SketchRef = exports.SketchesService = exports.SketchesServicePath = exports.SketchesError = void 0;
const application_error_1 = require("@theia/core/lib/common/application-error");
const uri_1 = require("@theia/core/lib/common/uri");
var SketchesError;
(function (SketchesError) {
    SketchesError.Codes = {
        NotFound: 5001,
    };
    SketchesError.NotFound = application_error_1.ApplicationError.declare(SketchesError.Codes.NotFound, (message, uri) => {
        return {
            message,
            data: { uri },
        };
    });
})(SketchesError = exports.SketchesError || (exports.SketchesError = {}));
exports.SketchesServicePath = '/services/sketches-service';
exports.SketchesService = Symbol('SketchesService');
var SketchRef;
(function (SketchRef) {
    function fromUri(uriLike) {
        const uri = typeof uriLike === 'string' ? new uri_1.default(uriLike) : uriLike;
        return {
            name: uri.path.base,
            uri: typeof uriLike === 'string' ? uriLike : uriLike.toString(),
        };
    }
    SketchRef.fromUri = fromUri;
})(SketchRef = exports.SketchRef || (exports.SketchRef = {}));
var Sketch;
(function (Sketch) {
    function is(arg) {
        return (!!arg &&
            'name' in arg &&
            'uri' in arg &&
            typeof arg.name === 'string' &&
            typeof arg.uri === 'string');
    }
    Sketch.is = is;
    let Extensions;
    (function (Extensions) {
        Extensions.MAIN = ['.ino', '.pde'];
        Extensions.SOURCE = ['.c', '.cpp', '.s'];
        Extensions.ADDITIONAL = [
            '.h',
            '.c',
            '.hpp',
            '.hh',
            '.cpp',
            '.S',
            '.json',
            '.md',
            '.adoc',
        ];
        Extensions.ALL = Array.from(new Set([...Extensions.MAIN, ...Extensions.SOURCE, ...Extensions.ADDITIONAL]));
    })(Extensions = Sketch.Extensions || (Sketch.Extensions = {}));
    function isInSketch(uri, sketch) {
        return uris(sketch).includes(typeof uri === 'string' ? uri : uri.toString());
    }
    Sketch.isInSketch = isInSketch;
    function isSketchFile(arg) {
        if (arg instanceof uri_1.default) {
            return isSketchFile(arg.toString());
        }
        return Extensions.MAIN.some((ext) => arg.endsWith(ext));
    }
    Sketch.isSketchFile = isSketchFile;
    function uris(sketch) {
        const { mainFileUri, otherSketchFileUris, additionalFileUris } = sketch;
        return [mainFileUri, ...otherSketchFileUris, ...additionalFileUris];
    }
    Sketch.uris = uris;
    const primitiveProps = ['name', 'uri', 'mainFileUri'];
    const arrayProps = [
        'additionalFileUris',
        'otherSketchFileUris',
        'rootFolderFileUris',
    ];
    function sameAs(left, right) {
        for (const prop of primitiveProps) {
            const leftValue = left[prop];
            const rightValue = right[prop];
            assertIsNotArray(leftValue, prop, left);
            assertIsNotArray(rightValue, prop, right);
            if (leftValue !== rightValue) {
                return false;
            }
        }
        for (const prop of arrayProps) {
            const leftValue = left[prop];
            const rightValue = right[prop];
            assertIsArray(leftValue, prop, left);
            assertIsArray(rightValue, prop, right);
            if (leftValue.length !== rightValue.length) {
                return false;
            }
        }
        for (const prop of arrayProps) {
            const leftValue = left[prop];
            const rightValue = right[prop];
            assertIsArray(leftValue, prop, left);
            assertIsArray(rightValue, prop, right);
            if (toSortedString(leftValue) !==
                toSortedString(rightValue)) {
                return false;
            }
        }
        return true;
    }
    Sketch.sameAs = sameAs;
    function toSortedString(array) {
        return array.slice().sort().join(',');
    }
    function assertIsNotArray(toTest, prop, object) {
        if (Array.isArray(toTest)) {
            throw new Error(`Expected a non-array type. Got: ${toTest}. Property was: ${prop}. Object was: ${JSON.stringify(object)}`);
        }
    }
    function assertIsArray(toTest, prop, object) {
        if (!Array.isArray(toTest)) {
            throw new Error(`Expected an array type. Got: ${toTest}. Property was: ${prop}. Object was: ${JSON.stringify(object)}`);
        }
    }
})(Sketch = exports.Sketch || (exports.Sketch = {}));
var SketchContainer;
(function (SketchContainer) {
    function create(label) {
        return {
            label,
            children: [],
            sketches: [],
        };
    }
    SketchContainer.create = create;
    function is(arg) {
        return (!!arg &&
            'label' in arg &&
            typeof arg.label === 'string' &&
            'children' in arg &&
            Array.isArray(arg.children) &&
            'sketches' in arg &&
            Array.isArray(arg.sketches));
    }
    SketchContainer.is = is;
    /**
     * `false` if the `container` recursively contains at least one sketch. Otherwise, `true`.
     */
    function isEmpty(container) {
        const hasSketch = (parent) => {
            if (parent.sketches.length ||
                parent.children.some((child) => hasSketch(child))) {
                return true;
            }
            return false;
        };
        return !hasSketch(container);
    }
    SketchContainer.isEmpty = isEmpty;
    function prune(container) {
        for (let i = container.children.length - 1; i >= 0; i--) {
            if (isEmpty(container.children[i])) {
                container.children.splice(i, 1);
            }
        }
        return container;
    }
    SketchContainer.prune = prune;
    function toArray(container) {
        const visit = (parent, toPushSketch) => {
            toPushSketch.push(...parent.sketches);
            parent.children.map((child) => visit(child, toPushSketch));
        };
        const sketches = [];
        visit(container, sketches);
        return sketches;
    }
    SketchContainer.toArray = toArray;
})(SketchContainer = exports.SketchContainer || (exports.SketchContainer = {}));
//# sourceMappingURL=sketches-service.js.map