"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.ElectronMainApplication = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const electron_1 = require("@theia/core/electron-shared/electron");
const child_process_1 = require("child_process");
const path_1 = require("path");
const fs = require("fs-extra");
const electron_token_1 = require("@theia/core/lib/electron-common/electron-token");
const electron_main_application_1 = require("@theia/core/lib/electron-main/electron-main-application");
const vscode_uri_1 = require("@theia/core/shared/vscode-uri");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const os = require("@theia/core/lib/common/os");
const electron_messages_1 = require("@theia/core/lib/electron-common/messaging/electron-messages");
const is_temp_sketch_1 = require("../../node/is-temp-sketch");
electron_1.app.commandLine.appendSwitch('disable-http-cache');
const WORKSPACES = 'workspaces';
/**
 * If the app is started with `--open-devtools` argument, the `Dev Tools` will be opened.
 */
const APP_STARTED_WITH_DEV_TOOLS = typeof process !== 'undefined' &&
    process.argv.indexOf('--open-devtools') !== -1;
/**
 * If the app is started with `--content-trace` argument, the `Dev Tools` will be opened and content tracing will start.
 */
const APP_STARTED_WITH_CONTENT_TRACE = typeof process !== 'undefined' &&
    process.argv.indexOf('--content-trace') !== -1;
let ElectronMainApplication = class ElectronMainApplication extends electron_main_application_1.ElectronMainApplication {
    constructor() {
        super(...arguments);
        this.startup = false;
        this.openFilePromise = new promise_util_1.Deferred();
        this.closedWorkspaces = [];
    }
    async start(config) {
        // Explicitly set the app name to have better menu items on macOS. ("About", "Hide", and "Quit")
        // See: https://github.com/electron-userland/electron-builder/issues/2468
        // Regression in Theia: https://github.com/eclipse-theia/theia/issues/8701
        electron_1.app.on('ready', () => electron_1.app.setName(config.applicationName));
        this.attachFileAssociations();
        this.useNativeWindowFrame = this.getTitleBarStyle(config) === 'native';
        this._config = config;
        this.hookApplicationEvents();
        const [port] = await Promise.all([this.startBackend(), electron_1.app.whenReady()]);
        this.startContentTracing();
        this._backendPort.resolve(port);
        await Promise.all([
            this.attachElectronSecurityToken(port),
            this.startContributions(),
        ]);
        return this.launch({
            secondInstance: false,
            argv: this.processArgv.getProcessArgvWithoutBin(process.argv),
            cwd: process.cwd(),
        });
    }
    startContentTracing() {
        if (!APP_STARTED_WITH_CONTENT_TRACE) {
            return;
        }
        if (!electron_1.app.isReady()) {
            throw new Error('Cannot start content tracing when the electron app is not ready.');
        }
        const defaultTraceCategories = [
            '-*',
            'devtools.timeline',
            'disabled-by-default-devtools.timeline',
            'disabled-by-default-devtools.timeline.frame',
            'toplevel',
            'blink.console',
            'disabled-by-default-devtools.timeline.stack',
            'disabled-by-default-v8.cpu_profile',
            'disabled-by-default-v8.cpu_profiler',
            'disabled-by-default-v8.cpu_profiler.hires',
        ];
        const traceOptions = {
            categoryFilter: defaultTraceCategories.join(','),
            traceOptions: 'record-until-full',
            options: 'sampling-frequency=10000',
        };
        (async () => {
            const appPath = electron_1.app.getAppPath();
            let traceFile;
            if (appPath) {
                const tracesPath = (0, path_1.join)(appPath, 'traces');
                await fs.promises.mkdir(tracesPath, { recursive: true });
                traceFile = (0, path_1.join)(tracesPath, `trace-${new Date().toISOString()}.trace`);
            }
            console.log('>>> Content tracing has started...');
            await electron_1.contentTracing.startRecording(traceOptions);
            await new Promise((resolve) => setTimeout(resolve, 10000));
            electron_1.contentTracing
                .stopRecording(traceFile)
                .then((out) => console.log(`<<< Content tracing has finished. The trace data was written to: ${out}.`));
        })();
    }
    attachFileAssociations() {
        // OSX: register open-file event
        if (os.isOSX) {
            electron_1.app.on('open-file', async (event, uri) => {
                event.preventDefault();
                if (uri.endsWith('.ino') && (await fs.pathExists(uri))) {
                    this.openFilePromise.reject();
                    await this.openSketch((0, path_1.dirname)(uri));
                }
            });
            setTimeout(() => this.openFilePromise.resolve(), 500);
        }
        else {
            this.openFilePromise.resolve();
        }
    }
    async isValidSketchPath(uri) {
        return typeof uri === 'string' && (await fs.pathExists(uri));
    }
    async launch(params) {
        try {
            // When running on MacOS, we either have to wait until
            // 1. The `open-file` command has been received by the app, rejecting the promise
            // 2. A short timeout resolves the promise automatically, falling back to the usual app launch
            await this.openFilePromise.promise;
        }
        catch (_a) {
            // Application has received the `open-file` event and will skip the default application launch
            return;
        }
        if (!os.isOSX && (await this.launchFromArgs(params))) {
            // Application has received a file in its arguments and will skip the default application launch
            return;
        }
        this.startup = true;
        const workspaces = this.electronStore.get(WORKSPACES);
        let useDefault = true;
        if (workspaces && workspaces.length > 0) {
            console.log(`Restoring workspace roots: ${workspaces.map(({ file }) => file)}`);
            for (const workspace of workspaces) {
                if (await this.isValidSketchPath(workspace.file)) {
                    if (this.isTempSketch.is(workspace.file)) {
                        console.info(`Skipped opening sketch. The sketch was detected as temporary. Workspace path: ${workspace.file}.`);
                        continue;
                    }
                    useDefault = false;
                    await this.openSketch(workspace);
                }
            }
        }
        this.startup = false;
        if (useDefault) {
            super.launch(params);
        }
    }
    async launchFromArgs(params) {
        // Copy to prevent manipulation of original array
        const argCopy = [...params.argv];
        let uri;
        for (const possibleUri of argCopy) {
            if (possibleUri.endsWith('.ino') &&
                (await this.isValidSketchPath(possibleUri))) {
                uri = possibleUri;
                break;
            }
        }
        if (uri) {
            await this.openSketch((0, path_1.dirname)(uri));
            return true;
        }
        return false;
    }
    async openSketch(workspace) {
        const options = await this.getLastWindowOptions();
        let file;
        if (typeof workspace === 'object') {
            options.x = workspace.x;
            options.y = workspace.y;
            options.width = workspace.width;
            options.height = workspace.height;
            options.isMaximized = workspace.isMaximized;
            options.isFullScreen = workspace.isFullScreen;
            file = workspace.file;
        }
        else {
            file = workspace;
        }
        const [uri, electronWindow] = await Promise.all([
            this.createWindowUri(),
            this.createWindow(options),
        ]);
        electronWindow.loadURL(uri.withFragment(encodeURI(file)).toString(true));
        return electronWindow;
    }
    avoidOverlap(options) {
        if (this.startup) {
            return options;
        }
        return super.avoidOverlap(options);
    }
    getTitleBarStyle(
    // eslint-disable-next-line @typescript-eslint/no-unused-vars
    _config) {
        return 'native';
    }
    hookApplicationEvents() {
        electron_1.app.on('will-quit', this.onWillQuit.bind(this));
        electron_1.app.on('second-instance', this.onSecondInstance.bind(this));
        electron_1.app.on('window-all-closed', this.onWindowAllClosed.bind(this));
        electron_1.ipcMain.on(electron_messages_1.Restart, ({ sender }) => {
            this.restart(sender.id);
        });
    }
    async onSecondInstance(event, argv, cwd) {
        if (!os.isOSX &&
            (await this.launchFromArgs({ cwd, argv, secondInstance: true }))) {
            // Application has received a file in its arguments
            return;
        }
        super.onSecondInstance(event, argv, cwd);
    }
    async createWindow(asyncOptions = this.getDefaultTheiaWindowOptions()) {
        const electronWindow = await super.createWindow(asyncOptions);
        if (APP_STARTED_WITH_DEV_TOOLS) {
            electronWindow.webContents.openDevTools();
        }
        this.attachListenersToWindow(electronWindow);
        if (this._firstWindowId === undefined) {
            this._firstWindowId = electronWindow.id;
        }
        return electronWindow;
    }
    getDefaultOptions() {
        const options = super.getDefaultOptions();
        if (!options.webPreferences) {
            options.webPreferences = {};
        }
        options.webPreferences.v8CacheOptions = 'bypassHeatCheck'; // TODO: verify this. VS Code use this V8 option.
        options.minWidth = 680;
        options.minHeight = 593;
        return options;
    }
    attachListenersToWindow(electronWindow) {
        electronWindow.webContents.on('new-window', (event, url, frameName, disposition, options) => {
            var _a, _b;
            if (frameName === 'serialPlotter') {
                event.preventDefault();
                Object.assign(options, {
                    width: 800,
                    minWidth: 620,
                    height: 500,
                    minHeight: 320,
                    x: 100,
                    y: 100,
                    webPreferences: {
                        devTools: true,
                        nativeWindowOpen: true,
                        openerId: electronWindow === null || electronWindow === void 0 ? void 0 : electronWindow.webContents.id,
                    },
                });
                event.newGuest = new electron_1.BrowserWindow(options);
                event.newGuest.setMenu(null);
                (_a = event.newGuest) === null || _a === void 0 ? void 0 : _a.on('closed', () => {
                    electronWindow === null || electronWindow === void 0 ? void 0 : electronWindow.webContents.send('CLOSE_CHILD_WINDOW');
                });
                (_b = event.newGuest) === null || _b === void 0 ? void 0 : _b.loadURL(url);
            }
        });
        this.attachClosedWorkspace(electronWindow);
    }
    async startBackend() {
        // Check if we should run everything as one process.
        const noBackendFork = process.argv.indexOf('--no-cluster') !== -1;
        // We cannot use the `process.cwd()` as the application project path (the location of the `package.json` in other words)
        // in a bundled electron application because it depends on the way we start it. For instance, on OS X, these are a differences:
        // https://github.com/eclipse-theia/theia/issues/3297#issuecomment-439172274
        process.env.THEIA_APP_PROJECT_PATH = this.globals.THEIA_APP_PROJECT_PATH;
        // Set the electron version for both the dev and the production mode. (https://github.com/eclipse-theia/theia/issues/3254)
        // Otherwise, the forked backend processes will not know that they're serving the electron frontend.
        process.env.THEIA_ELECTRON_VERSION = process.versions.electron;
        if (noBackendFork) {
            process.env[electron_token_1.ElectronSecurityToken] = JSON.stringify(this.electronSecurityToken);
            // The backend server main file is supposed to export a promise resolving with the port used by the http(s) server.
            const address = await require(this.globals
                .THEIA_BACKEND_MAIN_PATH);
            return address.port;
        }
        else {
            let args = this.processArgv.getProcessArgvWithoutBin();
            // https://github.com/eclipse-theia/theia/issues/8227
            if (process.platform === 'darwin') {
                // https://github.com/electron/electron/issues/3657
                // https://stackoverflow.com/questions/10242115/os-x-strange-psn-command-line-parameter-when-launched-from-finder#comment102377986_10242200
                // macOS appends an extra `-psn_0_someNumber` arg if a file is opened from Finder after downloading from the Internet.
                // "AppName" is an app downloaded from the Internet. Are you sure you want to open it?
                args = args.filter((arg) => !arg.startsWith('-psn'));
            }
            const backendProcess = (0, child_process_1.fork)(this.globals.THEIA_BACKEND_MAIN_PATH, args, await this.getForkOptions());
            console.log(`Starting backend process. PID: ${backendProcess.pid}`);
            return new Promise((resolve, reject) => {
                // The backend server main file is also supposed to send the resolved http(s) server port via IPC.
                backendProcess.on('message', (address) => {
                    resolve(address.port);
                });
                backendProcess.on('error', (error) => {
                    reject(error);
                });
                electron_1.app.on('quit', () => {
                    try {
                        // If we forked the process for the clusters, we need to manually terminate it.
                        // See: https://github.com/eclipse-theia/theia/issues/835
                        if (backendProcess.pid) {
                            process.kill(backendProcess.pid);
                        }
                    }
                    catch (e) {
                        if (e.code === 'ESRCH') {
                            console.log('Could not terminate the backend process. It was not running.');
                            return;
                        }
                        throw e;
                    }
                });
            });
        }
    }
    attachClosedWorkspace(window) {
        // Since the `before-quit` event is only fired when closing the *last* window
        // We need to keep track of recently closed windows/workspaces manually
        window.on('close', () => {
            const url = window.webContents.getURL();
            const workspace = vscode_uri_1.URI.parse(url).fragment;
            if (workspace) {
                const workspaceUri = vscode_uri_1.URI.file(workspace);
                const bounds = window.getNormalBounds();
                const now = Date.now();
                // Do not try to reopen the sketch if it was temp.
                // Unfortunately, IDE2 has two different logic of restoring recent sketches: the Theia default `recentworkspace.json` and there is the `recent-sketches.json`.
                const file = workspaceUri.fsPath;
                if (this.isTempSketch.is(file)) {
                    console.info(`Ignored marking workspace as a closed sketch. The sketch was detected as temporary. Workspace URI: ${workspaceUri.toString()}.`);
                    return;
                }
                console.log(`Marking workspace as a closed sketch. Workspace URI: ${workspaceUri.toString()}. Date: ${now}.`);
                this.closedWorkspaces.push(Object.assign(Object.assign({}, bounds), { isMaximized: window.isMaximized(), isFullScreen: window.isFullScreen(), file: workspaceUri.fsPath, time: now }));
            }
        });
    }
    onWillQuit(event) {
        // Only add workspaces which were closed within the last second (1000 milliseconds)
        const threshold = Date.now() - 1000;
        const visited = new Set();
        const workspaces = this.closedWorkspaces
            .filter((e) => {
            if (e.time < threshold) {
                console.log(`Skipped storing sketch as workspace root. Expected minimum threshold: <${threshold}>. Was: <${e.time}>.`);
                return false;
            }
            if (visited.has(e.file)) {
                console.log(`Skipped storing sketch as workspace root. Already visited: <${e.file}>.`);
                return false;
            }
            visited.add(e.file);
            console.log(`Storing the sketch as a workspace root: <${e.file}>.`);
            return true;
        })
            .sort((a, b) => a.file.localeCompare(b.file));
        this.electronStore.set(WORKSPACES, workspaces);
        console.log(`Stored workspaces roots: ${workspaces.map(({ file }) => file)}`);
        super.onWillQuit(event);
    }
    get browserWindows() {
        return Array.from(this.windows.values()).map(({ window }) => window);
    }
    get firstWindowId() {
        return this._firstWindowId;
    }
};
__decorate([
    (0, inversify_1.inject)(is_temp_sketch_1.IsTempSketch),
    __metadata("design:type", is_temp_sketch_1.IsTempSketch)
], ElectronMainApplication.prototype, "isTempSketch", void 0);
ElectronMainApplication = __decorate([
    (0, inversify_1.injectable)()
], ElectronMainApplication);
exports.ElectronMainApplication = ElectronMainApplication;
//# sourceMappingURL=electron-main-application.js.map