"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DaemonError = exports.ArduinoDaemonImpl = void 0;
const path_1 = require("path");
const fs_1 = require("fs");
const inversify_1 = require("@theia/core/shared/inversify");
const child_process_1 = require("child_process");
const file_uri_1 = require("@theia/core/lib/node/file-uri");
const logger_1 = require("@theia/core/lib/common/logger");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const disposable_1 = require("@theia/core/lib/common/disposable");
const event_1 = require("@theia/core/lib/common/event");
const environment_1 = require("@theia/application-package/lib/environment");
const env_variables_1 = require("@theia/core/lib/common/env-variables");
const protocol_1 = require("../common/protocol");
const cli_config_1 = require("./cli-config");
const exec_util_1 = require("./exec-util");
let ArduinoDaemonImpl = class ArduinoDaemonImpl {
    constructor() {
        this.toDispose = new disposable_1.DisposableCollection();
        this.onDaemonStartedEmitter = new event_1.Emitter();
        this.onDaemonStoppedEmitter = new event_1.Emitter();
        this._running = false;
        this._port = new promise_util_1.Deferred();
    }
    // Backend application lifecycle.
    onStart() {
        this.start(); // no await
    }
    // Daemon API
    async getPort() {
        return this._port.promise;
    }
    async tryGetPort() {
        if (this._running) {
            return this._port.promise;
        }
        return undefined;
    }
    async start() {
        try {
            this.toDispose.dispose(); // This will `kill` the previously started daemon process, if any.
            const cliPath = await this.getExecPath();
            this.onData(`Starting daemon from ${cliPath}...`);
            const { daemon, port } = await this.spawnDaemonProcess();
            // Watchdog process for terminating the daemon process when the backend app terminates.
            (0, child_process_1.spawn)(process.execPath, [
                (0, path_1.join)(__dirname, 'daemon-watcher.js'),
                String(process.pid),
                String(daemon.pid),
            ], {
                env: environment_1.environment.electron.runAsNodeEnv(),
                detached: true,
                stdio: 'ignore',
                windowsHide: true,
            }).unref();
            this.toDispose.pushAll([
                disposable_1.Disposable.create(() => daemon.kill()),
                disposable_1.Disposable.create(() => this.fireDaemonStopped()),
            ]);
            this.fireDaemonStarted(port);
            this.onData('Daemon is running.');
            return port;
        }
        catch (err) {
            return (0, promise_util_1.retry)(() => {
                this.onError(err);
                return this.start();
            }, 1000, 5);
        }
    }
    async stop() {
        this.toDispose.dispose();
    }
    async restart() {
        return this.start();
    }
    // Backend only daemon API
    get onDaemonStarted() {
        return this.onDaemonStartedEmitter.event;
    }
    get onDaemonStopped() {
        return this.onDaemonStoppedEmitter.event;
    }
    async getExecPath() {
        if (this._execPath) {
            return this._execPath;
        }
        this._execPath = await (0, exec_util_1.getExecPath)('arduino-cli', this.onError.bind(this));
        return this._execPath;
    }
    async getVersion() {
        const execPath = await this.getExecPath();
        const raw = await (0, exec_util_1.spawnCommand)(`"${execPath}"`, ['version', '--format', 'json'], this.onError.bind(this));
        try {
            // The CLI `Info` object: https://github.com/arduino/arduino-cli/blob/17d24eb901b1fdaa5a4ec7da3417e9e460f84007/version/version.go#L31-L34
            const { VersionString, Commit, Status } = JSON.parse(raw);
            return {
                version: VersionString,
                commit: Commit,
                status: Status,
            };
        }
        catch (_a) {
            return { version: raw, commit: raw };
        }
    }
    async getSpawnArgs() {
        const [configDirUri, debug] = await Promise.all([
            this.envVariablesServer.getConfigDirUri(),
            this.debugDaemon(),
        ]);
        const cliConfigPath = (0, path_1.join)(file_uri_1.FileUri.fsPath(configDirUri), cli_config_1.CLI_CONFIG);
        const args = [
            'daemon',
            '--format',
            'jsonmini',
            '--port',
            '0',
            '--config-file',
            `"${cliConfigPath}"`,
            '-v',
            '--log-format',
            'json',
        ];
        if (debug) {
            args.push('--debug');
        }
        return args;
    }
    async debugDaemon() {
        // Poor man's preferences on the backend. (https://github.com/arduino/arduino-ide/issues/1056#issuecomment-1153975064)
        const configDirUri = await this.envVariablesServer.getConfigDirUri();
        const configDirPath = file_uri_1.FileUri.fsPath(configDirUri);
        try {
            const raw = await fs_1.promises.readFile((0, path_1.join)(configDirPath, 'settings.json'), {
                encoding: 'utf8',
            });
            const json = this.tryParse(raw);
            if (json) {
                const value = json['arduino.cli.daemon.debug'];
                return typeof value === 'boolean' && !!value;
            }
            return false;
        }
        catch (error) {
            if ('code' in error && error.code === 'ENOENT') {
                return false;
            }
            throw error;
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    tryParse(raw) {
        try {
            return JSON.parse(raw);
        }
        catch (_a) {
            return undefined;
        }
    }
    async spawnDaemonProcess() {
        const [cliPath, args] = await Promise.all([
            this.getExecPath(),
            this.getSpawnArgs(),
        ]);
        const ready = new promise_util_1.Deferred();
        const options = { shell: true };
        const daemon = (0, child_process_1.spawn)(`"${cliPath}"`, args, options);
        // If the process exists right after the daemon gRPC server has started (due to an invalid port, unknown address, TCP port in use, etc.)
        // we have no idea about the root cause unless we sniff into the first data package and dispatch the logic on that. Note, we get a exit code 1.
        let grpcServerIsReady = false;
        daemon.stdout.on('data', (data) => {
            const message = data.toString();
            let port = '';
            let address = '';
            message
                .split('\n')
                .filter((line) => line.length)
                .forEach((line) => {
                try {
                    const parsedLine = JSON.parse(line);
                    if ('Port' in parsedLine) {
                        port = parsedLine.Port;
                    }
                    if ('IP' in parsedLine) {
                        address = parsedLine.IP;
                    }
                }
                catch (err) {
                    // ignore
                }
            });
            this.onData(message);
            if (!grpcServerIsReady) {
                const error = DaemonError.parse(message);
                if (error) {
                    ready.reject(error);
                    return;
                }
                if (port.length && address.length) {
                    grpcServerIsReady = true;
                    ready.resolve({ daemon, port });
                }
            }
        });
        daemon.stderr.on('data', (data) => {
            const message = data.toString();
            this.onData(data.toString());
            const error = DaemonError.parse(message);
            ready.reject(error ? error : new Error(data.toString().trim()));
        });
        daemon.on('exit', (code, signal) => {
            if (code === 0 || signal === 'SIGINT' || signal === 'SIGKILL') {
                this.onData('Daemon has stopped.');
            }
            else {
                this.onData(`Daemon exited with ${typeof code === 'undefined'
                    ? `signal '${signal}'`
                    : `exit code: ${code}`}.`);
            }
        });
        daemon.on('error', (error) => {
            this.onError(error);
            ready.reject(error);
        });
        return ready.promise;
    }
    fireDaemonStarted(port) {
        this._running = true;
        this._port.resolve(port);
        this.onDaemonStartedEmitter.fire(port);
        this.notificationService.notifyDaemonDidStart(port);
    }
    fireDaemonStopped() {
        if (!this._running) {
            return;
        }
        this._running = false;
        this._port.reject(); // Reject all pending.
        this._port = new promise_util_1.Deferred();
        this.onDaemonStoppedEmitter.fire();
        this.notificationService.notifyDaemonDidStop();
    }
    onData(message) {
        this.logger.info(message);
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    onError(error) {
        this.logger.error(error);
    }
};
__decorate([
    (0, inversify_1.inject)(logger_1.ILogger),
    (0, inversify_1.named)('daemon'),
    __metadata("design:type", Object)
], ArduinoDaemonImpl.prototype, "logger", void 0);
__decorate([
    (0, inversify_1.inject)(env_variables_1.EnvVariablesServer),
    __metadata("design:type", Object)
], ArduinoDaemonImpl.prototype, "envVariablesServer", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], ArduinoDaemonImpl.prototype, "notificationService", void 0);
ArduinoDaemonImpl = __decorate([
    (0, inversify_1.injectable)()
], ArduinoDaemonImpl);
exports.ArduinoDaemonImpl = ArduinoDaemonImpl;
class DaemonError extends Error {
    constructor(message, code, details) {
        super(message);
        this.code = code;
        this.details = details;
        Object.setPrototypeOf(this, DaemonError.prototype);
    }
}
exports.DaemonError = DaemonError;
(function (DaemonError) {
    DaemonError.ADDRESS_IN_USE = 0;
    DaemonError.UNKNOWN_ADDRESS = 2;
    DaemonError.INVALID_PORT = 4;
    DaemonError.UNKNOWN = 8;
    function parse(log) {
        const raw = log.toLocaleLowerCase();
        if (raw.includes('failed to listen')) {
            if (raw.includes('address already in use') ||
                (raw.includes('bind') &&
                    raw.includes('only one usage of each socket address'))) {
                return new DaemonError('Failed to listen on TCP port. Address already in use.', DaemonError.ADDRESS_IN_USE);
            }
            if (raw.includes('is unknown name') ||
                (raw.includes('tcp/') && raw.includes('is an invalid port'))) {
                return new DaemonError('Failed to listen on TCP port. Unknown address.', DaemonError.UNKNOWN_ADDRESS);
            }
            if (raw.includes('is an invalid port')) {
                return new DaemonError('Failed to listen on TCP port. Invalid port.', DaemonError.INVALID_PORT);
            }
        }
        // Based on the CLI logging: `failed to serve`, and  any other FATAL errors.
        // https://github.com/arduino/arduino-cli/blob/11abbee8a9f027d087d4230f266a87217677d423/cli/daemon/daemon.go#L89-L94
        if (raw.includes('failed to serve') &&
            (raw.includes('"fatal"') || raw.includes('fata'))) {
            return new DaemonError('Unexpected CLI start error.', DaemonError.UNKNOWN, log);
        }
        return undefined;
    }
    DaemonError.parse = parse;
})(DaemonError = exports.DaemonError || (exports.DaemonError = {}));
//# sourceMappingURL=arduino-daemon-impl.js.map