"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardDiscovery = void 0;
const disposable_1 = require("@theia/core/lib/common/disposable");
const event_1 = require("@theia/core/lib/common/event");
const logger_1 = require("@theia/core/lib/common/logger");
const objects_1 = require("@theia/core/lib/common/objects");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const inversify_1 = require("@theia/core/shared/inversify");
const vscode_languageserver_protocol_1 = require("@theia/core/shared/vscode-languageserver-protocol");
const uuid_1 = require("uuid");
const nls_1 = require("../common/nls");
const protocol_1 = require("../common/protocol");
const board_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/board_pb");
const core_client_provider_1 = require("./core-client-provider");
const service_error_1 = require("./service-error");
/**
 * Singleton service for tracking the available ports and board and broadcasting the
 * changes to all connected frontend instances.
 *
 * Unlike other services, this is not connection scoped.
 */
let BoardDiscovery = class BoardDiscovery extends core_client_provider_1.CoreClientAware {
    constructor() {
        super(...arguments);
        this.onStreamDidEndEmitter = new event_1.Emitter(); // sent from the CLI when the discovery process is killed for example after the indexes update and the core client re-initialization.
        this.onStreamDidCancelEmitter = new event_1.Emitter(); // when the watcher is canceled by the IDE2
        this.toDisposeOnStopWatch = new disposable_1.DisposableCollection();
        this.uploadInProgress = false;
        /**
         * Keys are the `address` of the ports.
         *
         * The `protocol` is ignored because the board detach event does not carry the protocol information,
         * just the address.
         * ```json
         * {
         *  "type": "remove",
         *  "address": "/dev/cu.usbmodem14101"
         * }
         * ```
         */
        this._availablePorts = {};
    }
    get availablePorts() {
        return this._availablePorts;
    }
    onStart() {
        this.start();
        this.onClientDidRefresh(() => this.restart());
    }
    async restart() {
        this.logger.info('restarting before stop');
        await this.stop();
        this.logger.info('restarting after stop');
        return this.start();
    }
    onStop() {
        this.stop();
    }
    async stop(restart = false) {
        this.logger.info('stop');
        if (this.stopping) {
            this.logger.info('stop already stopping');
            return this.stopping.promise;
        }
        if (!this.watching) {
            return;
        }
        this.stopping = new promise_util_1.Deferred();
        this.logger.info('>>> Stopping boards watcher...');
        return new Promise((resolve, reject) => {
            const timeout = this.createTimeout(10000, reject);
            const toDispose = new disposable_1.DisposableCollection();
            const waitForEvent = (event) => event(() => {
                var _a;
                this.logger.info('stop received event: either end or cancel');
                toDispose.dispose();
                (_a = this.stopping) === null || _a === void 0 ? void 0 : _a.resolve();
                this.stopping = undefined;
                this.logger.info('stop stopped');
                resolve();
                if (restart) {
                    this.start();
                }
            });
            toDispose.pushAll([
                timeout,
                waitForEvent(this.onStreamDidEndEmitter.event),
                waitForEvent(this.onStreamDidCancelEmitter.event),
            ]);
            this.logger.info('Canceling boards watcher...');
            this.toDisposeOnStopWatch.dispose();
        });
    }
    setUploadInProgress(uploadInProgress) {
        this.uploadInProgress = uploadInProgress;
    }
    createTimeout(after, onTimeout) {
        const timer = setTimeout(() => onTimeout(new Error(`Timed out after ${after} ms.`)), after);
        return vscode_languageserver_protocol_1.Disposable.create(() => clearTimeout(timer));
    }
    async requestStartWatch(req, duplex) {
        return new Promise((resolve, reject) => {
            if (!duplex.write(req, (err) => {
                if (err) {
                    reject(err);
                    return;
                }
            })) {
                duplex.once('drain', resolve);
            }
            else {
                process.nextTick(resolve);
            }
        });
    }
    async createWrapper(client) {
        if (this.wrapper) {
            throw new Error(`Duplex was already set.`);
        }
        const stream = client
            .boardListWatch()
            .on('end', () => {
            this.logger.info('received end');
            this.onStreamDidEndEmitter.fire();
        })
            .on('error', (error) => {
            this.logger.info('error received');
            if (service_error_1.ServiceError.isCancel(error)) {
                this.logger.info('cancel error received!');
                this.onStreamDidCancelEmitter.fire();
            }
            else {
                this.logger.error('Unexpected error occurred during the boards discovery.', error);
                // TODO: terminate? restart? reject?
            }
        });
        const wrapper = {
            stream,
            uuid: (0, uuid_1.v4)(),
            dispose: () => {
                this.logger.info('disposing requesting cancel');
                // Cancelling the stream will kill the discovery `builtin:mdns-discovery process`.
                // The client (this class) will receive a `{"eventType":"quit","error":""}` response from the CLI.
                stream.cancel();
                this.logger.info('disposing canceled');
                this.wrapper = undefined;
            },
        };
        this.toDisposeOnStopWatch.pushAll([
            wrapper,
            vscode_languageserver_protocol_1.Disposable.create(() => {
                var _a;
                (_a = this.watching) === null || _a === void 0 ? void 0 : _a.reject(new Error(`Stopping watcher.`));
                this.watching = undefined;
            }),
        ]);
        return wrapper;
    }
    toJson(arg) {
        let object = undefined;
        if (arg instanceof board_pb_1.BoardListWatchRequest) {
            object = board_pb_1.BoardListWatchRequest.toObject(false, arg);
        }
        else if (arg instanceof board_pb_1.BoardListWatchResponse) {
            object = board_pb_1.BoardListWatchResponse.toObject(false, arg);
        }
        else {
            throw new Error(`Unhandled object type: ${arg}`);
        }
        return JSON.stringify(object);
    }
    async start() {
        this.logger.info('start');
        if (this.stopping) {
            this.logger.info('start is stopping wait');
            await this.stopping.promise;
            this.logger.info('start stopped');
        }
        if (this.watching) {
            this.logger.info('start already watching');
            return this.watching.promise;
        }
        this.watching = new promise_util_1.Deferred();
        this.logger.info('start new deferred');
        const { client, instance } = await this.coreClient;
        const wrapper = await this.createWrapper(client);
        wrapper.stream.on('data', (resp) => this.onBoardListWatchResponse(resp));
        this.logger.info('start request start watch');
        await this.requestStartWatch(new board_pb_1.BoardListWatchRequest().setInstance(instance), wrapper.stream);
        this.logger.info('start requested start watch');
        this.watching.resolve();
        this.logger.info('start resolved watching');
    }
    // XXX: make this `protected` and override for tests if IDE2 wants to mock events from the CLI.
    onBoardListWatchResponse(resp) {
        this.logger.info(this.toJson(resp));
        const eventType = EventType.parse(resp.getEventType());
        if (eventType === EventType.Quit) {
            this.logger.info('quit received');
            this.stop();
            return;
        }
        const detectedPort = resp.getPort();
        if (detectedPort) {
            const { port, boards } = this.fromRpc(detectedPort);
            if (!port) {
                if (!!boards.length) {
                    console.warn(`Could not detect the port, but unexpectedly received discovered boards. This is most likely a bug! Response was: ${this.toJson(resp)}`);
                }
                return;
            }
            const oldState = (0, objects_1.deepClone)(this._availablePorts);
            const newState = (0, objects_1.deepClone)(this._availablePorts);
            const key = protocol_1.Port.keyOf(port);
            if (eventType === EventType.Add) {
                if (newState[key]) {
                    const [, knownBoards] = newState[key];
                    this.logger.warn(`Port '${protocol_1.Port.toString(port)}' was already available. Known boards before override: ${JSON.stringify(knownBoards)}`);
                }
                newState[key] = [port, boards];
            }
            else if (eventType === EventType.Remove) {
                if (!newState[key]) {
                    this.logger.warn(`Port '${protocol_1.Port.toString(port)}' was not available. Skipping`);
                    return;
                }
                delete newState[key];
            }
            const event = {
                oldState: Object.assign({}, protocol_1.AvailablePorts.split(oldState)),
                newState: Object.assign({}, protocol_1.AvailablePorts.split(newState)),
                uploadInProgress: this.uploadInProgress,
            };
            this._availablePorts = newState;
            this.notificationService.notifyAttachedBoardsDidChange(event);
        }
    }
    fromRpc(detectedPort) {
        const rpcPort = detectedPort.getPort();
        const port = rpcPort && this.fromRpcPort(rpcPort);
        const boards = detectedPort.getMatchingBoardsList().map((board) => ({
            fqbn: board.getFqbn(),
            name: board.getName() || nls_1.Unknown,
            port,
        }));
        return {
            boards,
            port,
        };
    }
    fromRpcPort(rpcPort) {
        const port = {
            address: rpcPort.getAddress(),
            addressLabel: rpcPort.getLabel(),
            protocol: rpcPort.getProtocol(),
            protocolLabel: rpcPort.getProtocolLabel(),
            properties: protocol_1.Port.Properties.create(rpcPort.getPropertiesMap().toObject()),
        };
        return port;
    }
};
__decorate([
    (0, inversify_1.inject)(logger_1.ILogger),
    (0, inversify_1.named)('discovery-log'),
    __metadata("design:type", Object)
], BoardDiscovery.prototype, "logger", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], BoardDiscovery.prototype, "notificationService", void 0);
BoardDiscovery = __decorate([
    (0, inversify_1.injectable)()
], BoardDiscovery);
exports.BoardDiscovery = BoardDiscovery;
var EventType;
(function (EventType) {
    EventType[EventType["Add"] = 0] = "Add";
    EventType[EventType["Remove"] = 1] = "Remove";
    EventType[EventType["Quit"] = 2] = "Quit";
})(EventType || (EventType = {}));
(function (EventType) {
    function parse(type) {
        const normalizedType = type.toLowerCase();
        switch (normalizedType) {
            case 'add':
                return EventType.Add;
            case 'remove':
                return EventType.Remove;
            case 'quit':
                return EventType.Quit;
            default:
                throw new Error(`Unexpected 'BoardListWatchResponse' event type: '${type}.'`);
        }
    }
    EventType.parse = parse;
})(EventType || (EventType = {}));
//# sourceMappingURL=board-discovery.js.map