"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.BoardsServiceImpl = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const logger_1 = require("@theia/core/lib/common/logger");
const objects_1 = require("@theia/core/lib/common/objects");
const protocol_1 = require("../common/protocol");
const core_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/core_pb");
const board_discovery_1 = require("./board-discovery");
const core_client_provider_1 = require("./core-client-provider");
const board_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/board_pb");
const upload_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/upload_pb");
const grpc_progressible_1 = require("./grpc-progressible");
const service_error_1 = require("./service-error");
const common_1 = require("@theia/core/lib/common");
let BoardsServiceImpl = class BoardsServiceImpl extends core_client_provider_1.CoreClientAware {
    async getState() {
        return this.boardDiscovery.availablePorts;
    }
    async getBoardDetails(options) {
        const coreClient = await this.coreClient;
        const { client, instance } = coreClient;
        const { fqbn } = options;
        const detailsReq = new board_pb_1.BoardDetailsRequest();
        detailsReq.setInstance(instance);
        detailsReq.setFqbn(fqbn);
        const detailsResp = await new Promise((resolve, reject) => client.boardDetails(detailsReq, (err, resp) => {
            if (err) {
                if (isMissingPlatformError(err)) {
                    resolve(undefined);
                    return;
                }
                reject(err);
                return;
            }
            resolve(resp);
        }));
        if (!detailsResp) {
            return undefined;
        }
        const debuggingSupported = detailsResp.getDebuggingSupported();
        const requiredTools = detailsResp.getToolsDependenciesList().map((t) => ({
            name: t.getName(),
            packager: t.getPackager(),
            version: t.getVersion(),
        }));
        const configOptions = detailsResp.getConfigOptionsList().map((c) => ({
            label: c.getOptionLabel(),
            option: c.getOption(),
            values: c.getValuesList().map((v) => ({
                value: v.getValue(),
                label: v.getValueLabel(),
                selected: v.getSelected(),
            })),
        }));
        const listReq = new upload_pb_1.ListProgrammersAvailableForUploadRequest();
        listReq.setInstance(instance);
        listReq.setFqbn(fqbn);
        const listResp = await new Promise((resolve, reject) => client.listProgrammersAvailableForUpload(listReq, (err, resp) => {
            if (err) {
                reject(err);
                return;
            }
            resolve(resp);
        }));
        const programmers = listResp.getProgrammersList().map((p) => ({
            id: p.getId(),
            name: p.getName(),
            platform: p.getPlatform(),
        }));
        let VID = 'N/A';
        let PID = 'N/A';
        const prop = detailsResp
            .getIdentificationPropertiesList()
            .map((item) => item.getPropertiesMap())
            .find(objects_1.notEmpty);
        if (prop) {
            VID = prop.get('vid') || '';
            PID = prop.get('pid') || '';
        }
        return {
            fqbn,
            requiredTools,
            configOptions,
            programmers,
            debuggingSupported,
            VID,
            PID,
        };
    }
    async getBoardPackage(options) {
        const { id: expectedId } = options;
        if (!expectedId) {
            return undefined;
        }
        const packages = await this.search({ query: expectedId });
        return packages.find(({ id }) => id === expectedId);
    }
    async getContainerBoardPackage(options) {
        const { fqbn: expectedFqbn } = options;
        if (!expectedFqbn) {
            return undefined;
        }
        const packages = await this.search({});
        return packages.find(({ boards }) => boards.some(({ fqbn }) => fqbn === expectedFqbn));
    }
    async searchBoards({ query, }) {
        const { instance, client } = await this.coreClient;
        const req = new board_pb_1.BoardSearchRequest();
        req.setSearchArgs(query || '');
        req.setInstance(instance);
        const boards = await new Promise((resolve, reject) => {
            client.boardSearch(req, (error, resp) => {
                if (error) {
                    reject(error);
                    return;
                }
                const boards = [];
                for (const board of resp.getBoardsList()) {
                    const platform = board.getPlatform();
                    if (platform) {
                        boards.push({
                            name: board.getName(),
                            fqbn: board.getFqbn(),
                            packageId: platform.getId(),
                            packageName: platform.getName(),
                            manuallyInstalled: platform.getManuallyInstalled(),
                        });
                    }
                }
                resolve(boards);
            });
        });
        return boards;
    }
    async getBoardUserFields(options) {
        const coreClient = await this.coreClient;
        const { client, instance } = coreClient;
        const req = new upload_pb_1.SupportedUserFieldsRequest();
        req.setInstance(instance);
        req.setFqbn(options.fqbn);
        req.setProtocol(options.protocol);
        const resp = await new Promise((resolve, reject) => {
            client.supportedUserFields(req, (err, resp) => {
                if (err) {
                    if (isMissingPlatformError(err)) {
                        resolve(undefined);
                        return;
                    }
                    reject(err);
                    return;
                }
                resolve(resp);
            });
        });
        if (!resp) {
            return [];
        }
        return resp.getUserFieldsList().map((e) => ({
            toolId: e.getToolId(),
            name: e.getName(),
            label: e.getLabel(),
            secret: e.getSecret(),
            value: '',
        }));
    }
    async search(options) {
        const coreClient = await this.coreClient;
        const { client, instance } = coreClient;
        const installedPlatformsReq = new core_pb_1.PlatformListRequest();
        installedPlatformsReq.setInstance(instance);
        const installedPlatformsResp = await new Promise((resolve, reject) => {
            client.platformList(installedPlatformsReq, (err, resp) => {
                !!err ? reject(err) : resolve(resp);
            });
        });
        const installedPlatforms = installedPlatformsResp.getInstalledPlatformsList();
        const req = new core_pb_1.PlatformSearchRequest();
        req.setSearchArgs(options.query || '');
        req.setAllVersions(true);
        req.setInstance(instance);
        const resp = await new Promise((resolve, reject) => {
            client.platformSearch(req, (err, resp) => {
                !!err ? reject(err) : resolve(resp);
            });
        });
        const packages = new Map();
        const toPackage = (platform) => {
            let installedVersion;
            const matchingPlatform = installedPlatforms.find((ip) => ip.getId() === platform.getId());
            if (!!matchingPlatform) {
                installedVersion = matchingPlatform.getInstalled();
            }
            return {
                id: platform.getId(),
                name: platform.getName(),
                author: platform.getMaintainer(),
                availableVersions: [platform.getLatest()],
                description: platform
                    .getBoardsList()
                    .map((b) => b.getName())
                    .join(', '),
                installable: true,
                types: platform.getTypeList(),
                deprecated: platform.getDeprecated(),
                summary: common_1.nls.localize('arduino/component/boardsIncluded', 'Boards included in this package:'),
                installedVersion,
                boards: platform
                    .getBoardsList()
                    .map((b) => ({ name: b.getName(), fqbn: b.getFqbn() })),
                moreInfoLink: platform.getWebsite(),
            };
        };
        // We must group the cores by ID, and sort platforms by, first the installed version, then version alphabetical order.
        // Otherwise we lose the FQBN information.
        const groupedById = new Map();
        for (const platform of resp.getSearchOutputList()) {
            const id = platform.getId();
            const idGroup = groupedById.get(id);
            if (idGroup) {
                idGroup.push(platform);
            }
            else {
                groupedById.set(id, [platform]);
            }
        }
        const installedAwareVersionComparator = (left, right) => {
            // XXX: we cannot rely on `platform.getInstalled()`, it is always an empty string.
            const leftInstalled = !!installedPlatforms.find((ip) => ip.getId() === left.getId() && ip.getInstalled() === left.getLatest());
            const rightInstalled = !!installedPlatforms.find((ip) => ip.getId() === right.getId() &&
                ip.getInstalled() === right.getLatest());
            if (leftInstalled && !rightInstalled) {
                return -1;
            }
            if (!leftInstalled && rightInstalled) {
                return 1;
            }
            const invertedVersionComparator = protocol_1.Installable.Version.COMPARATOR(left.getLatest(), right.getLatest()) *
                -1;
            // Higher version comes first.
            return invertedVersionComparator;
        };
        for (const value of groupedById.values()) {
            value.sort(installedAwareVersionComparator);
        }
        for (const value of groupedById.values()) {
            for (const platform of value) {
                const id = platform.getId();
                const pkg = packages.get(id);
                if (pkg) {
                    pkg.availableVersions.push(platform.getLatest());
                    pkg.availableVersions.sort(protocol_1.Installable.Version.COMPARATOR).reverse();
                }
                else {
                    packages.set(id, toPackage(platform));
                }
            }
        }
        const filter = this.typePredicate(options);
        return [...packages.values()].filter(filter);
    }
    typePredicate(options) {
        const { type } = options;
        if (!type || type === 'All') {
            return () => true;
        }
        switch (options.type) {
            case 'Updatable':
                return protocol_1.Installable.Updateable;
            case 'Arduino':
            case 'Partner':
            case 'Arduino@Heart':
            case 'Contributed':
            case 'Arduino Certified':
                return ({ types }) => !!types && (types === null || types === void 0 ? void 0 : types.includes(type));
            default:
                throw new Error(`Unhandled type: ${options.type}`);
        }
    }
    async install(options) {
        const item = options.item;
        const version = !!options.version
            ? options.version
            : item.availableVersions[0];
        const coreClient = await this.coreClient;
        const { client, instance } = coreClient;
        const [platform, architecture] = item.id.split(':');
        const req = new core_pb_1.PlatformInstallRequest();
        req.setInstance(instance);
        req.setArchitecture(architecture);
        req.setPlatformPackage(platform);
        req.setVersion(version);
        req.setNoOverwrite(Boolean(options.noOverwrite));
        console.info('>>> Starting boards package installation...', item);
        // stop the board discovery
        await this.boardDiscovery.stop();
        const resp = client.platformInstall(req);
        resp.on('data', grpc_progressible_1.ExecuteWithProgress.createDataCallback({
            progressId: options.progressId,
            responseService: this.responseService,
        }));
        await new Promise((resolve, reject) => {
            resp.on('end', () => {
                this.boardDiscovery.start(); // TODO: remove discovery dependency from boards service. See https://github.com/arduino/arduino-ide/pull/1107 why this is here.
                resolve();
            });
            resp.on('error', (error) => {
                this.responseService.appendToOutput({
                    chunk: `Failed to install platform: ${item.id}.\n`,
                });
                this.responseService.appendToOutput({
                    chunk: `${error.toString()}\n`,
                });
                reject(error);
            });
        });
        const items = await this.search({});
        const updated = items.find((other) => protocol_1.BoardsPackage.equals(other, item)) || item;
        this.notificationService.notifyPlatformDidInstall({ item: updated });
        console.info('<<< Boards package installation done.', item);
    }
    async uninstall(options) {
        const { item, progressId } = options;
        const coreClient = await this.coreClient;
        const { client, instance } = coreClient;
        const [platform, architecture] = item.id.split(':');
        const req = new core_pb_1.PlatformUninstallRequest();
        req.setInstance(instance);
        req.setArchitecture(architecture);
        req.setPlatformPackage(platform);
        console.info('>>> Starting boards package uninstallation...', item);
        // stop the board discovery
        await this.boardDiscovery.stop();
        const resp = client.platformUninstall(req);
        resp.on('data', grpc_progressible_1.ExecuteWithProgress.createDataCallback({
            progressId,
            responseService: this.responseService,
        }));
        await new Promise((resolve, reject) => {
            resp.on('end', () => {
                this.boardDiscovery.start(); // TODO: remove discovery dependency from boards service. See https://github.com/arduino/arduino-ide/pull/1107 why this is here.
                resolve();
            });
            resp.on('error', reject);
        });
        // Here, unlike at `install` we send out the argument `item`. Otherwise, we would not know about the board FQBN.
        this.notificationService.notifyPlatformDidUninstall({ item });
        console.info('<<< Boards package uninstallation done.', item);
    }
};
__decorate([
    (0, inversify_1.inject)(logger_1.ILogger),
    __metadata("design:type", Object)
], BoardsServiceImpl.prototype, "logger", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.ResponseService),
    __metadata("design:type", Object)
], BoardsServiceImpl.prototype, "responseService", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], BoardsServiceImpl.prototype, "notificationService", void 0);
__decorate([
    (0, inversify_1.inject)(board_discovery_1.BoardDiscovery),
    __metadata("design:type", board_discovery_1.BoardDiscovery)
], BoardsServiceImpl.prototype, "boardDiscovery", void 0);
BoardsServiceImpl = __decorate([
    (0, inversify_1.injectable)()
], BoardsServiceImpl);
exports.BoardsServiceImpl = BoardsServiceImpl;
function isMissingPlatformError(error) {
    if (service_error_1.ServiceError.is(error)) {
        const message = error.details;
        // TODO: check gRPC status code? `9 FAILED_PRECONDITION` (https://grpc.github.io/grpc/core/md_doc_statuscodes.html)
        // When installing a 3rd party core that depends on a missing Arduino core.
        // https://github.com/arduino/arduino-cli/issues/954
        if (message.includes('missing platform release') &&
            message.includes('referenced by board')) {
            return true;
        }
        // When the platform is not installed.
        if (message.includes('platform') && message.includes('not installed')) {
            return true;
        }
        // It's a hack to handle https://github.com/arduino/arduino-cli/issues/1262 gracefully.
        if (message.includes('unknown package')) {
            return true;
        }
    }
    return false;
}
//# sourceMappingURL=boards-service-impl.js.map