"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.tryParseError = exports.OutputSource = void 0;
const objects_1 = require("@theia/core/lib/common/objects");
const nls_1 = require("@theia/core/lib/common/nls");
const file_uri_1 = require("@theia/core/lib/node/file-uri");
const vscode_languageserver_protocol_1 = require("@theia/core/shared/vscode-languageserver-protocol");
const protocol_1 = require("../common/protocol");
const sketches_service_1 = require("../common/protocol/sketches-service");
var OutputSource;
(function (OutputSource) {
    function content(source) {
        const { content } = source;
        return typeof content === 'string'
            ? content
            : Buffer.concat(content).toString('utf8');
    }
    OutputSource.content = content;
})(OutputSource = exports.OutputSource || (exports.OutputSource = {}));
function tryParseError(source) {
    const { sketch } = source;
    const content = OutputSource.content(source);
    if (sketch) {
        return tryParse(content)
            .map(remapErrorMessages)
            .filter(isLocationInSketch(sketch))
            .map(toErrorInfo)
            .reduce((acc, curr) => {
            const existingRef = acc.find((candidate) => protocol_1.CoreError.ErrorLocationRef.equals(candidate, curr));
            if (existingRef) {
                existingRef.rangesInOutput.push(...curr.rangesInOutput);
            }
            else {
                acc.push(curr);
            }
            return acc;
        }, []);
    }
    return [];
}
exports.tryParseError = tryParseError;
var ParseResult;
(function (ParseResult) {
    function keyOf(result) {
        /**
         * The CLI compiler might return with the same error multiple times. This is the key function for the distinct set calculation.
         */
        return JSON.stringify(result);
    }
    ParseResult.keyOf = keyOf;
})(ParseResult || (ParseResult = {}));
function isLocationInSketch(sketch) {
    return (result) => {
        const uri = file_uri_1.FileUri.create(result.path).toString();
        if (!sketches_service_1.Sketch.isInSketch(uri, sketch)) {
            console.warn(`URI <${uri}> is not contained in sketch: <${JSON.stringify(sketch)}>`);
            return false;
        }
        return true;
    };
}
function toErrorInfo({ error, message, path, line, column, rangeInOutput, }) {
    return {
        message: error,
        details: message,
        location: {
            uri: file_uri_1.FileUri.create(path).toString(),
            range: range(line, column),
        },
        rangesInOutput: rangeInOutput ? [rangeInOutput] : [],
    };
}
function range(line, column) {
    const start = vscode_languageserver_protocol_1.Position.create(line - 1, typeof column === 'number' ? column - 1 : 0);
    return {
        start,
        end: start,
    };
}
function tryParse(content) {
    var _a;
    // Shamelessly stolen from the Java IDE: https://github.com/arduino/Arduino/blob/43b0818f7fa8073301db1b80ac832b7b7596b828/arduino-core/src/cc/arduino/Compiler.java#L137
    const re = new RegExp('(.+\\.\\w+):(\\d+)(:\\d+)*:\\s*((fatal)?\\s*error:\\s*)(.*)\\s*', 'gm');
    return Array.from((_a = content.matchAll(re)) !== null && _a !== void 0 ? _a : [])
        .map((match) => {
        const { index: start } = match;
        const [, path, rawLine, rawColumn, errorPrefix, , error] = match.map((match) => (match ? match.trim() : match));
        const line = Number.parseInt(rawLine, 10);
        if (!Number.isInteger(line)) {
            console.warn(`Could not parse line number. Raw input: <${rawLine}>, parsed integer: <${line}>.`);
            return undefined;
        }
        let column = undefined;
        if (rawColumn) {
            const normalizedRawColumn = rawColumn.slice(-1); // trims the leading colon => `:3` will be `3`
            column = Number.parseInt(normalizedRawColumn, 10);
            if (!Number.isInteger(column)) {
                console.warn(`Could not parse column number. Raw input: <${normalizedRawColumn}>, parsed integer: <${column}>.`);
            }
        }
        const rangeInOutput = findRangeInOutput(start, { path, rawLine, rawColumn }, content);
        return {
            path,
            line,
            column,
            errorPrefix,
            error,
            rangeInOutput,
        };
    })
        .filter(objects_1.notEmpty);
}
/**
 * Converts cryptic and legacy error messages to nice ones. Taken from the Java IDE.
 */
function remapErrorMessages(result) {
    const knownError = KnownErrors[result.error];
    if (!knownError) {
        return result;
    }
    const { message, error } = knownError;
    return Object.assign(Object.assign(Object.assign({}, result), (message && { message })), (error && { error }));
}
// Based on the Java IDE: https://github.com/arduino/Arduino/blob/43b0818f7fa8073301db1b80ac832b7b7596b828/arduino-core/src/cc/arduino/Compiler.java#L528-L578
const KnownErrors = {
    "'Mouse' was not declared in this scope": {
        error: nls_1.nls.localize('arduino/cli-error-parser/mouseError', "'Mouse' not found. Does your sketch include the line '#include <Mouse.h>'?"),
    },
    "'Keyboard' was not declared in this scope": {
        error: nls_1.nls.localize('arduino/cli-error-parser/keyboardError', "'Keyboard' not found. Does your sketch include the line '#include <Keyboard.h>'?"),
    },
};
function findRangeInOutput(startIndex, groups, content // TODO? lines: string[]? can this code break line on `\n`? const lines = content.split(/\r?\n/) ?? [];
) {
    if (startIndex === undefined) {
        return undefined;
    }
    // /path/to/location/Sketch/Sketch.ino:36:42
    const offset = groups.path.length +
        ':'.length +
        groups.rawLine.length +
        (groups.rawColumn ? groups.rawColumn.length : 0);
    const start = toPosition(startIndex, content);
    if (!start) {
        return undefined;
    }
    const end = toPosition(startIndex + offset, content);
    if (!end) {
        return undefined;
    }
    return { start, end };
}
function toPosition(offset, content) {
    let line = 0;
    let character = 0;
    const length = content.length;
    for (let i = 0; i < length; i++) {
        const c = content.charAt(i);
        if (i === offset) {
            return { line, character };
        }
        if (c === '\n') {
            line++;
            character = 0;
        }
        else {
            character++;
        }
    }
    return undefined;
}
//# sourceMappingURL=cli-error-parser.js.map