"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.CoreClientAware = exports.CoreClientProvider = void 0;
const path_1 = require("path");
const grpc = require("@grpc/grpc-js");
const inversify_1 = require("@theia/core/shared/inversify");
const event_1 = require("@theia/core/lib/common/event");
const commands_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/commands_pb");
const commandsGrpcPb = require("./cli-protocol/cc/arduino/cli/commands/v1/commands_grpc_pb");
const protocol_1 = require("../common/protocol");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const status_pb_1 = require("./cli-protocol/google/rpc/status_pb");
const config_service_impl_1 = require("./config-service-impl");
const arduino_daemon_impl_1 = require("./arduino-daemon-impl");
const disposable_1 = require("@theia/core/lib/common/disposable");
const vscode_languageserver_protocol_1 = require("@theia/core/shared/vscode-languageserver-protocol");
const grpc_progressible_1 = require("./grpc-progressible");
const service_error_1 = require("./service-error");
let CoreClientProvider = class CoreClientProvider {
    constructor() {
        this.ready = new promise_util_1.Deferred();
        this.toDisposeBeforeCreate = new disposable_1.DisposableCollection();
        this.toDisposeAfterDidCreate = new disposable_1.DisposableCollection();
        this.onClientReadyEmitter = new event_1.Emitter();
        this.onClientReady = this.onClientReadyEmitter.event;
        this.onClientDidRefreshEmitter = new event_1.Emitter();
    }
    init() {
        this.daemon.tryGetPort().then((port) => {
            if (port) {
                this.create(port);
            }
        });
        this.daemon.onDaemonStarted((port) => this.create(port));
        this.daemon.onDaemonStopped(() => this.closeClient());
        this.configService.onConfigChange(() => this.refreshIndexes());
    }
    get tryGetClient() {
        return this._client;
    }
    get client() {
        const client = this.tryGetClient;
        if (client) {
            return Promise.resolve(client);
        }
        if (!this.pending) {
            this.pending = new promise_util_1.Deferred();
            this.toDisposeAfterDidCreate.pushAll([
                vscode_languageserver_protocol_1.Disposable.create(() => (this.pending = undefined)),
                this.onClientReady((client) => {
                    var _a;
                    (_a = this.pending) === null || _a === void 0 ? void 0 : _a.resolve(client);
                    this.toDisposeAfterDidCreate.dispose();
                }),
            ]);
        }
        return this.pending.promise;
    }
    get onClientDidRefresh() {
        return this.onClientDidRefreshEmitter.event;
    }
    /**
     * Encapsulates both the gRPC core client creation (`CreateRequest`) and initialization (`InitRequest`).
     */
    async create(port) {
        this.closeClient();
        const address = this.address(port);
        const client = await this.createClient(address);
        this.toDisposeBeforeCreate.pushAll([
            vscode_languageserver_protocol_1.Disposable.create(() => client.client.close()),
            vscode_languageserver_protocol_1.Disposable.create(() => {
                this.ready.reject(new Error(`Disposed. Creating a new gRPC core client on address ${address}.`));
                this.ready = new promise_util_1.Deferred();
            }),
        ]);
        await this.initInstanceWithFallback(client);
        setTimeout(async () => this.refreshIndexes(), 10000); // Update the indexes asynchronously
        return this.useClient(client);
    }
    /**
     * By default, calling this method is equivalent to the `initInstance(Client)` call.
     * When the IDE2 starts and one of the followings is missing,
     * the IDE2 must run the index update before the core client initialization:
     *
     *  - primary package index (`#directories.data/package_index.json`),
     *  - library index (`#directories.data/library_index.json`),
     *  - built-in tools (`builtin:serial-discovery` or `builtin:mdns-discovery`)
     *
     * This method detects such errors and runs an index update before initializing the client.
     * The index update will fail if the 3rd URLs list contains an invalid URL,
     * and the IDE2 will be [non-functional](https://github.com/arduino/arduino-ide/issues/1084). Since the CLI [cannot update only the primary package index]((https://github.com/arduino/arduino-cli/issues/1788)), IDE2 does its dirty solution.
     */
    async initInstanceWithFallback(client) {
        try {
            await this.initInstance(client);
        }
        catch (err) {
            if (err instanceof IndexUpdateRequiredBeforeInitError) {
                console.error('The primary packages indexes are missing. Running indexes update before initializing the core gRPC client', err.message);
                await this.updateIndexes(client); // TODO: this should run without the 3rd party URLs
                await this.initInstance(client);
                console.info(`Downloaded the primary package indexes, and successfully initialized the core gRPC client.`);
            }
            else {
                console.error('Error occurred while initializing the core gRPC client provider', err);
                throw err;
            }
        }
    }
    useClient(client) {
        this._client = client;
        this.onClientReadyEmitter.fire(this._client);
        return this._client;
    }
    closeClient() {
        return this.toDisposeBeforeCreate.dispose();
    }
    async createClient(address) {
        // https://github.com/agreatfool/grpc_tools_node_protoc_ts/blob/master/doc/grpcjs_support.md#usage
        const ArduinoCoreServiceClient = grpc.makeClientConstructor(
        // @ts-expect-error: ignore
        commandsGrpcPb['cc.arduino.cli.commands.v1.ArduinoCoreService'], 'ArduinoCoreServiceService'
        // eslint-disable-next-line @typescript-eslint/no-explicit-any
        );
        const client = new ArduinoCoreServiceClient(address, grpc.credentials.createInsecure(), this.channelOptions);
        const instance = await new Promise((resolve, reject) => {
            client.create(new commands_pb_1.CreateRequest(), (err, resp) => {
                if (err) {
                    reject(err);
                    return;
                }
                const instance = resp.getInstance();
                if (!instance) {
                    reject(new Error('`CreateResponse` was OK, but the retrieved `instance` was `undefined`.'));
                    return;
                }
                resolve(instance);
            });
        });
        return { instance, client };
    }
    async initInstance({ client, instance, }) {
        return new Promise((resolve, reject) => {
            const errors = [];
            client
                .init(new commands_pb_1.InitRequest().setInstance(instance))
                .on('data', (resp) => {
                // XXX: The CLI never sends `initProgress`, it's always `error` or nothing. Is this a CLI bug?
                // According to the gRPC API, the CLI should send either a `TaskProgress` or a `DownloadProgress`, but it does not.
                const error = resp.getError();
                if (error) {
                    const { code, message } = status_pb_1.Status.toObject(false, error);
                    console.error(`Detected an error response during the gRPC core client initialization: code: ${code}, message: ${message}`);
                    errors.push(error);
                }
            })
                .on('error', reject)
                .on('end', () => {
                const error = this.evaluateErrorStatus(errors);
                if (error) {
                    reject(error);
                    return;
                }
                resolve();
            });
        });
    }
    evaluateErrorStatus(status) {
        const { cliConfiguration } = this.configService;
        if (!cliConfiguration) {
            // If the CLI config is not available, do not even try to guess what went wrong.
            return new Error(`Could not read the CLI configuration file.`);
        }
        return isIndexUpdateRequiredBeforeInit(status, cliConfiguration); // put future error matching here
    }
    /**
     * Updates all indexes and runs an init to [reload the indexes](https://github.com/arduino/arduino-cli/pull/1274#issue-866154638).
     */
    async refreshIndexes() {
        const client = this._client;
        if (client) {
            const progressHandler = this.createProgressHandler();
            try {
                await this.updateIndexes(client, progressHandler);
                await this.initInstance(client);
                // notify clients about the index update only after the client has been "re-initialized" and the new content is available.
                progressHandler.reportEnd();
                this.onClientDidRefreshEmitter.fire(client);
            }
            catch (err) {
                console.error('Failed to update indexes', err);
                progressHandler.reportError(service_error_1.ServiceError.is(err) ? err.details : String(err));
            }
        }
    }
    async updateIndexes(client, progressHandler) {
        await Promise.all([
            this.updateIndex(client, progressHandler),
            this.updateLibraryIndex(client, progressHandler),
        ]);
    }
    async updateIndex(client, progressHandler) {
        return this.doUpdateIndex(() => client.client.updateIndex(new commands_pb_1.UpdateIndexRequest().setInstance(client.instance)), progressHandler, 'platform-index');
    }
    async updateLibraryIndex(client, progressHandler) {
        return this.doUpdateIndex(() => client.client.updateLibrariesIndex(new commands_pb_1.UpdateLibrariesIndexRequest().setInstance(client.instance)), progressHandler, 'library-index');
    }
    async doUpdateIndex(responseProvider, progressHandler, task) {
        const progressId = progressHandler === null || progressHandler === void 0 ? void 0 : progressHandler.progressId;
        return (0, promise_util_1.retry)(() => new Promise((resolve, reject) => {
            responseProvider()
                .on('data', grpc_progressible_1.ExecuteWithProgress.createDataCallback({
                responseService: {
                    appendToOutput: ({ chunk: message }) => {
                        console.log(`core-client-provider${task ? ` [${task}]` : ''}`, message);
                        progressHandler === null || progressHandler === void 0 ? void 0 : progressHandler.reportProgress(message);
                    },
                },
                progressId,
            }))
                .on('error', reject)
                .on('end', resolve);
        }), 50, 3);
    }
    createProgressHandler() {
        var _a, _b, _c, _d;
        const additionalUrlsCount = (_d = (_c = (_b = (_a = this.configService.cliConfiguration) === null || _a === void 0 ? void 0 : _a.board_manager) === null || _b === void 0 ? void 0 : _b.additional_urls) === null || _c === void 0 ? void 0 : _c.length) !== null && _d !== void 0 ? _d : 0;
        return new grpc_progressible_1.IndexesUpdateProgressHandler(additionalUrlsCount, (progressMessage) => this.notificationService.notifyIndexUpdateDidProgress(progressMessage), ({ progressId, message }) => this.notificationService.notifyIndexUpdateDidFail({
            progressId,
            message,
        }), (progressId) => this.notificationService.notifyIndexWillUpdate(progressId), (progressId) => this.notificationService.notifyIndexDidUpdate(progressId));
    }
    address(port) {
        return `localhost:${port}`;
    }
    get channelOptions() {
        return {
            'grpc.max_send_message_length': 512 * 1024 * 1024,
            'grpc.max_receive_message_length': 512 * 1024 * 1024,
            'grpc.primary_user_agent': `arduino-ide/${this.version}`,
        };
    }
    get version() {
        if (this._version) {
            return this._version;
        }
        const json = require('../../package.json');
        if ('version' in json) {
            this._version = json.version;
        }
        if (!this._version) {
            this._version = '0.0.0';
        }
        return this._version;
    }
};
__decorate([
    (0, inversify_1.inject)(arduino_daemon_impl_1.ArduinoDaemonImpl),
    __metadata("design:type", arduino_daemon_impl_1.ArduinoDaemonImpl)
], CoreClientProvider.prototype, "daemon", void 0);
__decorate([
    (0, inversify_1.inject)(config_service_impl_1.ConfigServiceImpl),
    __metadata("design:type", config_service_impl_1.ConfigServiceImpl)
], CoreClientProvider.prototype, "configService", void 0);
__decorate([
    (0, inversify_1.inject)(protocol_1.NotificationServiceServer),
    __metadata("design:type", Object)
], CoreClientProvider.prototype, "notificationService", void 0);
__decorate([
    (0, inversify_1.postConstruct)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], CoreClientProvider.prototype, "init", null);
CoreClientProvider = __decorate([
    (0, inversify_1.injectable)()
], CoreClientProvider);
exports.CoreClientProvider = CoreClientProvider;
/**
 * Sugar for making the gRPC core client available for the concrete service classes.
 */
let CoreClientAware = class CoreClientAware {
    /**
     * Returns with a promise that resolves when the core client is initialized and ready.
     */
    get coreClient() {
        return this.coreClientProvider.client;
    }
    get onClientDidRefresh() {
        return this.coreClientProvider.onClientDidRefresh;
    }
};
__decorate([
    (0, inversify_1.inject)(CoreClientProvider),
    __metadata("design:type", CoreClientProvider)
], CoreClientAware.prototype, "coreClientProvider", void 0);
CoreClientAware = __decorate([
    (0, inversify_1.injectable)()
], CoreClientAware);
exports.CoreClientAware = CoreClientAware;
class IndexUpdateRequiredBeforeInitError extends Error {
    constructor(causes) {
        super(`The index of the cores and libraries must be updated before initializing the core gRPC client.
The following problems were detected during the gRPC client initialization:
${causes
            .map(({ code, message }) => ` - code: ${code}, message: ${message}`)
            .join('\n')}
`);
        Object.setPrototypeOf(this, IndexUpdateRequiredBeforeInitError.prototype);
        if (!causes.length) {
            throw new Error(`expected non-empty 'causes'`);
        }
    }
}
function isIndexUpdateRequiredBeforeInit(status, cliConfig) {
    const causes = status
        .filter((s) => IndexUpdateRequiredBeforeInit.map((predicate) => predicate(s, cliConfig)).some(Boolean))
        .map((s) => status_pb_1.Status.toObject(false, s));
    return causes.length
        ? new IndexUpdateRequiredBeforeInitError(causes)
        : undefined;
}
const IndexUpdateRequiredBeforeInit = [
    isPackageIndexMissingStatus,
    isDiscoveryNotFoundStatus,
];
function isPackageIndexMissingStatus(status, { directories: { data } }) {
    const predicate = ({ message }) => message.includes('loading json index file') &&
        (message.includes((0, path_1.join)(data, 'package_index.json')) ||
            message.includes((0, path_1.join)(data, 'library_index.json')));
    // https://github.com/arduino/arduino-cli/blob/f0245bc2da6a56fccea7b2c9ea09e85fdcc52cb8/arduino/cores/packagemanager/package_manager.go#L247
    return evaluate(status, predicate);
}
function isDiscoveryNotFoundStatus(status) {
    const predicate = ({ message }) => message.includes('discovery') &&
        (message.includes('not found') || message.includes('not installed'));
    // https://github.com/arduino/arduino-cli/blob/f0245bc2da6a56fccea7b2c9ea09e85fdcc52cb8/arduino/cores/packagemanager/loader.go#L740
    // https://github.com/arduino/arduino-cli/blob/f0245bc2da6a56fccea7b2c9ea09e85fdcc52cb8/arduino/cores/packagemanager/loader.go#L744
    return evaluate(status, predicate);
}
function evaluate(subject, predicate) {
    const status = status_pb_1.Status.toObject(false, subject);
    return predicate(status);
}
//# sourceMappingURL=core-client-provider.js.map