"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.IndexesUpdateProgressHandler = exports.ExecuteWithProgress = void 0;
const uuid_1 = require("uuid");
const commands_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/commands_pb");
const compile_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/compile_pb");
const core_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/core_pb");
const lib_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/lib_pb");
const upload_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/upload_pb");
var LibraryProgressResponse;
(function (LibraryProgressResponse) {
    function is(response) {
        return (response instanceof lib_pb_1.LibraryInstallResponse ||
            response instanceof lib_pb_1.LibraryUninstallResponse ||
            response instanceof lib_pb_1.ZipLibraryInstallResponse);
    }
    LibraryProgressResponse.is = is;
    function workUnit(response) {
        return Object.assign({ task: response.getTaskProgress() }, (response instanceof lib_pb_1.LibraryInstallResponse && {
            download: response.getProgress(),
        }));
    }
    LibraryProgressResponse.workUnit = workUnit;
})(LibraryProgressResponse || (LibraryProgressResponse = {}));
var PlatformProgressResponse;
(function (PlatformProgressResponse) {
    function is(response) {
        return (response instanceof core_pb_1.PlatformInstallResponse ||
            response instanceof core_pb_1.PlatformUninstallResponse);
    }
    PlatformProgressResponse.is = is;
    function workUnit(response) {
        return Object.assign({ task: response.getTaskProgress() }, (response instanceof core_pb_1.PlatformInstallResponse && {
            download: response.getProgress(),
        }));
    }
    PlatformProgressResponse.workUnit = workUnit;
})(PlatformProgressResponse || (PlatformProgressResponse = {}));
var IndexProgressResponse;
(function (IndexProgressResponse) {
    function is(response) {
        return (response instanceof commands_pb_1.UpdateIndexResponse ||
            response instanceof commands_pb_1.UpdateLibrariesIndexResponse ||
            response instanceof commands_pb_1.UpdateCoreLibrariesIndexResponse // not used by the IDE2 but available for full typings compatibility
        );
    }
    IndexProgressResponse.is = is;
    function workUnit(response) {
        return { download: response.getDownloadProgress() };
    }
    IndexProgressResponse.workUnit = workUnit;
})(IndexProgressResponse || (IndexProgressResponse = {}));
var IndefiniteProgressResponse;
(function (IndefiniteProgressResponse) {
    function is(response) {
        return (response instanceof upload_pb_1.UploadResponse ||
            response instanceof upload_pb_1.UploadUsingProgrammerResponse ||
            response instanceof upload_pb_1.BurnBootloaderResponse);
    }
    IndefiniteProgressResponse.is = is;
})(IndefiniteProgressResponse || (IndefiniteProgressResponse = {}));
var DefiniteProgressResponse;
(function (DefiniteProgressResponse) {
    function is(response) {
        return response instanceof compile_pb_1.CompileResponse;
    }
    DefiniteProgressResponse.is = is;
})(DefiniteProgressResponse || (DefiniteProgressResponse = {}));
var CoreProgressResponse;
(function (CoreProgressResponse) {
    function is(response) {
        return (DefiniteProgressResponse.is(response) ||
            IndefiniteProgressResponse.is(response));
    }
    CoreProgressResponse.is = is;
    function workUnit(response) {
        if (DefiniteProgressResponse.is(response)) {
            return { task: response.getProgress() };
        }
        return UnitOfWork.Unknown;
    }
    CoreProgressResponse.workUnit = workUnit;
})(CoreProgressResponse || (CoreProgressResponse = {}));
var UnitOfWork;
(function (UnitOfWork) {
    UnitOfWork.Unknown = {};
})(UnitOfWork || (UnitOfWork = {}));
/**
 * It's solely a dev thing. Flip it to `true` if you want to debug the progress from the CLI responses.
 */
const DEBUG = false;
var ExecuteWithProgress;
(function (ExecuteWithProgress) {
    function createDataCallback({ responseService, progressId, }) {
        const uuid = (0, uuid_1.v4)();
        let localFile = '';
        let localTotalSize = Number.NaN;
        return (response) => {
            var _a, _b, _c, _d, _e, _f, _g;
            if (DEBUG) {
                const json = toJson(response);
                if (json) {
                    console.log(`Progress response [${uuid}]: ${json}`);
                }
            }
            const unitOfWork = resolve(response);
            const { task, download } = unitOfWork;
            if (!download && !task) {
                // report a fake unknown progress.
                if (unitOfWork === UnitOfWork.Unknown && progressId) {
                    if (progressId) {
                        (_a = responseService.reportProgress) === null || _a === void 0 ? void 0 : _a.call(responseService, {
                            progressId,
                            message: '',
                            work: { done: Number.NaN, total: Number.NaN },
                        });
                    }
                    return;
                }
                if (DEBUG) {
                    // This is still an API error from the CLI, but IDE2 ignores it.
                    // Technically, it does not cause an error, but could mess up the progress reporting.
                    // See an example of an empty object `{}` repose here: https://github.com/arduino/arduino-ide/issues/906#issuecomment-1171145630.
                    console.warn("Implementation error. Neither 'download' nor 'task' is available.");
                }
                return;
            }
            if (task && download) {
                throw new Error("Implementation error. Both 'download' and 'task' are available.");
            }
            if (task) {
                const message = task.getName() || task.getMessage();
                const percent = task.getPercent();
                if (message) {
                    if (progressId) {
                        (_b = responseService.reportProgress) === null || _b === void 0 ? void 0 : _b.call(responseService, {
                            progressId,
                            message,
                            work: { done: Number.NaN, total: Number.NaN },
                        });
                    }
                    (_c = responseService.appendToOutput) === null || _c === void 0 ? void 0 : _c.call(responseService, { chunk: `${message}\n` });
                }
                else if (percent) {
                    if (progressId) {
                        (_d = responseService.reportProgress) === null || _d === void 0 ? void 0 : _d.call(responseService, {
                            progressId,
                            message,
                            work: { done: percent, total: 100 },
                        });
                    }
                }
            }
            else if (download) {
                if (download.getFile() && !localFile) {
                    localFile = download.getFile();
                }
                if (download.getTotalSize() > 0 && Number.isNaN(localTotalSize)) {
                    localTotalSize = download.getTotalSize();
                }
                // This happens only once per file download.
                if (download.getTotalSize() && localFile) {
                    (_e = responseService.appendToOutput) === null || _e === void 0 ? void 0 : _e.call(responseService, { chunk: `${localFile}\n` });
                }
                if (progressId && localFile) {
                    let work = undefined;
                    if (download.getDownloaded() > 0 && !Number.isNaN(localTotalSize)) {
                        work = {
                            total: localTotalSize,
                            done: download.getDownloaded(),
                        };
                    }
                    (_f = responseService.reportProgress) === null || _f === void 0 ? void 0 : _f.call(responseService, {
                        progressId,
                        message: `Downloading ${localFile}`,
                        work,
                    });
                }
                if (download.getCompleted()) {
                    // Discard local state.
                    if (progressId && !Number.isNaN(localTotalSize)) {
                        (_g = responseService.reportProgress) === null || _g === void 0 ? void 0 : _g.call(responseService, {
                            progressId,
                            message: '',
                            work: { done: Number.NaN, total: Number.NaN },
                        });
                    }
                    localFile = '';
                    localTotalSize = Number.NaN;
                }
            }
        };
    }
    ExecuteWithProgress.createDataCallback = createDataCallback;
    function resolve(response) {
        if (LibraryProgressResponse.is(response)) {
            return LibraryProgressResponse.workUnit(response);
        }
        else if (PlatformProgressResponse.is(response)) {
            return PlatformProgressResponse.workUnit(response);
        }
        else if (IndexProgressResponse.is(response)) {
            return IndexProgressResponse.workUnit(response);
        }
        else if (CoreProgressResponse.is(response)) {
            return CoreProgressResponse.workUnit(response);
        }
        console.warn('Unhandled gRPC response', response);
        return {};
    }
    function toJson(response) {
        let object = undefined;
        if (response instanceof lib_pb_1.LibraryInstallResponse) {
            object = lib_pb_1.LibraryInstallResponse.toObject(false, response);
        }
        else if (response instanceof lib_pb_1.LibraryUninstallResponse) {
            object = lib_pb_1.LibraryUninstallResponse.toObject(false, response);
        }
        else if (response instanceof lib_pb_1.ZipLibraryInstallResponse) {
            object = lib_pb_1.ZipLibraryInstallResponse.toObject(false, response);
        }
        else if (response instanceof core_pb_1.PlatformInstallResponse) {
            object = core_pb_1.PlatformInstallResponse.toObject(false, response);
        }
        else if (response instanceof core_pb_1.PlatformUninstallResponse) {
            object = core_pb_1.PlatformUninstallResponse.toObject(false, response);
        }
        else if (response instanceof commands_pb_1.UpdateIndexResponse) {
            object = commands_pb_1.UpdateIndexResponse.toObject(false, response);
        }
        else if (response instanceof commands_pb_1.UpdateLibrariesIndexResponse) {
            object = commands_pb_1.UpdateLibrariesIndexResponse.toObject(false, response);
        }
        else if (response instanceof commands_pb_1.UpdateCoreLibrariesIndexResponse) {
            object = commands_pb_1.UpdateCoreLibrariesIndexResponse.toObject(false, response);
        }
        else if (response instanceof compile_pb_1.CompileResponse) {
            object = compile_pb_1.CompileResponse.toObject(false, response);
        }
        if (!object) {
            console.warn('Unhandled gRPC response', response);
            return undefined;
        }
        return JSON.stringify(object);
    }
})(ExecuteWithProgress = exports.ExecuteWithProgress || (exports.ExecuteWithProgress = {}));
class IndexesUpdateProgressHandler {
    constructor(additionalUrlsCount, onProgress, onError, onStart, onEnd) {
        var _a;
        this.onProgress = onProgress;
        this.onError = onError;
        this.onStart = onStart;
        this.onEnd = onEnd;
        this.done = 0;
        this.progressId = (0, uuid_1.v4)();
        this.total = IndexesUpdateProgressHandler.total(additionalUrlsCount);
        // Note: at this point, the IDE2 backend might not have any connected clients, so this notification is not delivered to anywhere
        // Hence, clients must handle gracefully when no `willUpdate` is received before any `didProgress`.
        (_a = this.onStart) === null || _a === void 0 ? void 0 : _a.call(this, this.progressId);
    }
    reportEnd() {
        var _a;
        (_a = this.onEnd) === null || _a === void 0 ? void 0 : _a.call(this, this.progressId);
    }
    reportProgress(message) {
        this.onProgress({
            message,
            progressId: this.progressId,
            work: { total: this.total, done: ++this.done },
        });
    }
    reportError(message) {
        var _a;
        (_a = this.onError) === null || _a === void 0 ? void 0 : _a.call(this, { progressId: this.progressId, message });
    }
    static total(additionalUrlsCount) {
        // +1 for the `package_index.tar.bz2` when updating the platform index.
        const totalPlatformIndexCount = additionalUrlsCount + 1;
        // The `library_index.json.gz` and `library_index.json.sig` when running the library index update.
        const totalLibraryIndexCount = 2;
        // +1 for the `initInstance` call after the index update (`reportEnd`)
        return totalPlatformIndexCount + totalLibraryIndexCount + 1;
    }
}
exports.IndexesUpdateProgressHandler = IndexesUpdateProgressHandler;
//# sourceMappingURL=grpc-progressible.js.map