"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __param = (this && this.__param) || function (paramIndex, decorator) {
    return function (target, key) { decorator(target, key, paramIndex); }
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.MonitorService = exports.MonitorServiceName = void 0;
const core_1 = require("@theia/core");
const inversify_1 = require("@theia/core/shared/inversify");
const protocol_1 = require("../common/protocol");
const monitor_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/monitor_pb");
const core_client_provider_1 = require("./core-client-provider");
const web_socket_provider_1 = require("./web-socket/web-socket-provider");
const port_pb_1 = require("arduino-ide-extension/src/node/cli-protocol/cc/arduino/cli/commands/v1/port_pb");
const monitor_settings_provider_1 = require("./monitor-settings/monitor-settings-provider");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const monitor_service_factory_1 = require("./monitor-service-factory");
exports.MonitorServiceName = 'monitor-service';
const MAX_WRITE_TO_STREAM_TRIES = 10;
const WRITE_TO_STREAM_TIMEOUT_MS = 30000;
let MonitorService = class MonitorService extends core_client_provider_1.CoreClientAware {
    constructor(options) {
        super();
        // Settings used by the currently running pluggable monitor.
        // They can be freely modified while running.
        this.settings = {};
        // List of messages received from the running pluggable monitor.
        // These are flushed from time to time to the frontend.
        this.messages = [];
        // Used to notify that the monitor is being disposed
        this.onDisposeEmitter = new core_1.Emitter();
        this.onDispose = this.onDisposeEmitter.event;
        this._initialized = new promise_util_1.Deferred();
        this.board = options.board;
        this.port = options.port;
        this.monitorID = options.monitorID;
    }
    init() {
        this.onWSClientsNumberChanged =
            this.webSocketProvider.onClientsNumberChanged(async (clients) => {
                if (clients === 0) {
                    // There are no more clients that want to receive
                    // data from this monitor, we can freely close
                    // and dispose it.
                    this.dispose();
                    return;
                }
                this.updateClientsSettings(this.settings);
            });
        this.portMonitorSettings(this.port.protocol, this.board.fqbn).then(async (settings) => {
            this.settings = Object.assign(Object.assign({}, this.settings), { pluggableMonitorSettings: await this.monitorSettingsProvider.getSettings(this.monitorID, settings) });
            this._initialized.resolve();
        });
    }
    get initialized() {
        return this._initialized.promise;
    }
    getWebsocketAddressPort() {
        return this.webSocketProvider.getAddress().port;
    }
    dispose() {
        var _a;
        this.stop();
        this.onDisposeEmitter.fire();
        (_a = this.onWSClientsNumberChanged) === null || _a === void 0 ? void 0 : _a.dispose();
    }
    /**
     * isStarted is used to know if the currently running pluggable monitor is started.
     * @returns true if pluggable monitor communication duplex is open,
     * false in all other cases.
     */
    isStarted() {
        return !!this.duplex;
    }
    /**
     * Start and connects a monitor using currently set board and port.
     * If a monitor is already started or board fqbn, port address and/or protocol
     * are missing nothing happens.
     * @returns a status to verify connection has been established.
     */
    async start() {
        var _a, _b, _c, _d, _e, _f, _g, _h, _j, _k, _l;
        if (((_a = this.creating) === null || _a === void 0 ? void 0 : _a.state) === 'unresolved')
            return this.creating.promise;
        this.creating = new promise_util_1.Deferred();
        if (this.duplex) {
            this.updateClientsSettings({
                monitorUISettings: { connected: true, serialPort: this.port.address },
            });
            this.creating.resolve(protocol_1.Status.ALREADY_CONNECTED);
            return this.creating.promise;
        }
        if (!((_b = this.board) === null || _b === void 0 ? void 0 : _b.fqbn) || !((_c = this.port) === null || _c === void 0 ? void 0 : _c.address) || !((_d = this.port) === null || _d === void 0 ? void 0 : _d.protocol)) {
            this.updateClientsSettings({ monitorUISettings: { connected: false } });
            this.creating.resolve(protocol_1.Status.CONFIG_MISSING);
            return this.creating.promise;
        }
        this.logger.info('starting monitor');
        // get default monitor settings from the CLI
        const defaultSettings = await this.portMonitorSettings(this.port.protocol, this.board.fqbn);
        // get actual settings from the settings provider
        this.settings = Object.assign(Object.assign({}, this.settings), { pluggableMonitorSettings: Object.assign(Object.assign({}, this.settings.pluggableMonitorSettings), (await this.monitorSettingsProvider.getSettings(this.monitorID, defaultSettings))) });
        const coreClient = await this.coreClient;
        const { instance } = coreClient;
        const monitorRequest = new monitor_pb_1.MonitorRequest();
        monitorRequest.setInstance(instance);
        if ((_e = this.board) === null || _e === void 0 ? void 0 : _e.fqbn) {
            monitorRequest.setFqbn(this.board.fqbn);
        }
        if (((_f = this.port) === null || _f === void 0 ? void 0 : _f.address) && ((_g = this.port) === null || _g === void 0 ? void 0 : _g.protocol)) {
            const rpcPort = new port_pb_1.Port();
            rpcPort.setAddress(this.port.address);
            rpcPort.setProtocol(this.port.protocol);
            monitorRequest.setPort(rpcPort);
        }
        const config = new monitor_pb_1.MonitorPortConfiguration();
        for (const id in this.settings.pluggableMonitorSettings) {
            const s = new monitor_pb_1.MonitorPortSetting();
            s.setSettingId(id);
            s.setValue(this.settings.pluggableMonitorSettings[id].selectedValue);
            config.addSettings(s);
        }
        monitorRequest.setPortConfiguration(config);
        const wroteToStreamSuccessfully = await this.pollWriteToStream(monitorRequest);
        if (wroteToStreamSuccessfully) {
            this.startMessagesHandlers();
            this.logger.info(`started monitor to ${(_h = this.port) === null || _h === void 0 ? void 0 : _h.address} using ${(_j = this.port) === null || _j === void 0 ? void 0 : _j.protocol}`);
            this.updateClientsSettings({
                monitorUISettings: { connected: true, serialPort: this.port.address },
            });
            this.creating.resolve(protocol_1.Status.OK);
            return this.creating.promise;
        }
        else {
            this.logger.warn(`failed starting monitor to ${(_k = this.port) === null || _k === void 0 ? void 0 : _k.address} using ${(_l = this.port) === null || _l === void 0 ? void 0 : _l.protocol}`);
            this.creating.resolve(protocol_1.Status.NOT_CONNECTED);
            return this.creating.promise;
        }
    }
    async createDuplex() {
        const coreClient = await this.coreClient;
        return coreClient.client.monitor();
    }
    setDuplexHandlers(duplex, additionalHandlers) {
        // default handlers
        duplex
            .on('close', () => {
            var _a, _b;
            this.duplex = null;
            this.updateClientsSettings({
                monitorUISettings: { connected: false },
            });
            this.logger.info(`monitor to ${(_a = this.port) === null || _a === void 0 ? void 0 : _a.address} using ${(_b = this.port) === null || _b === void 0 ? void 0 : _b.protocol} closed by client`);
        })
            .on('end', () => {
            var _a, _b;
            this.duplex = null;
            this.updateClientsSettings({
                monitorUISettings: { connected: false },
            });
            this.logger.info(`monitor to ${(_a = this.port) === null || _a === void 0 ? void 0 : _a.address} using ${(_b = this.port) === null || _b === void 0 ? void 0 : _b.protocol} closed by server`);
        });
        for (const handler of additionalHandlers) {
            duplex.on(handler.key, handler.callback);
        }
    }
    pollWriteToStream(request) {
        let attemptsRemaining = MAX_WRITE_TO_STREAM_TRIES;
        const writeTimeoutMs = WRITE_TO_STREAM_TIMEOUT_MS;
        const createWriteToStreamExecutor = (duplex) => (resolve, reject) => {
            const resolvingDuplexHandlers = [
                {
                    key: 'error',
                    callback: async (err) => {
                        this.logger.error(err);
                        resolve(false);
                        // TODO
                        // this.theiaFEClient?.notifyError()
                    },
                },
                {
                    key: 'data',
                    callback: async (monitorResponse) => {
                        if (monitorResponse.getError()) {
                            // TODO: Maybe disconnect
                            this.logger.error(monitorResponse.getError());
                            return;
                        }
                        if (monitorResponse.getSuccess()) {
                            resolve(true);
                            return;
                        }
                        const data = monitorResponse.getRxData();
                        const message = typeof data === 'string'
                            ? data
                            : new TextDecoder('utf8').decode(data);
                        this.messages.push(...splitLines(message));
                    },
                },
            ];
            this.setDuplexHandlers(duplex, resolvingDuplexHandlers);
            setTimeout(() => {
                reject();
            }, writeTimeoutMs);
            duplex.write(request);
        };
        const pollWriteToStream = new Promise((resolve) => {
            const startPolling = async () => {
                // here we create a new duplex but we don't yet
                // set "this.duplex", nor do we use "this.duplex" in our poll
                // as duplex 'end' / 'close' events (which we do not "await")
                // will set "this.duplex" to null
                const createdDuplex = await this.createDuplex();
                let pollingIsSuccessful;
                // attempt a "writeToStream" and "await" CLI response: success (true) or error (false)
                // if we get neither within WRITE_TO_STREAM_TIMEOUT_MS or an error we get undefined
                try {
                    const writeToStream = createWriteToStreamExecutor(createdDuplex);
                    pollingIsSuccessful = await new Promise(writeToStream);
                }
                catch (error) {
                    this.logger.error(error);
                }
                // CLI confirmed port opened successfully
                if (pollingIsSuccessful) {
                    this.duplex = createdDuplex;
                    resolve(true);
                    return;
                }
                // if "pollingIsSuccessful" is false
                // the CLI gave us an error, lets try again
                // after waiting 2 seconds if we've not already
                // reached MAX_WRITE_TO_STREAM_TRIES
                if (pollingIsSuccessful === false) {
                    attemptsRemaining -= 1;
                    if (attemptsRemaining > 0) {
                        setTimeout(startPolling, 2000);
                        return;
                    }
                    else {
                        resolve(false);
                        return;
                    }
                }
                // "pollingIsSuccessful" remains undefined:
                // we got no response from the CLI within 30 seconds
                // resolve to false and end the duplex connection
                resolve(false);
                createdDuplex.end();
                return;
            };
            startPolling();
        });
        return pollWriteToStream;
    }
    /**
     * Pauses the currently running monitor, it still closes the gRPC connection
     * with the underlying monitor process but it doesn't stop the message handlers
     * currently running.
     * This is mainly used to handle upload with the board/port combination
     * the monitor is listening to.
     * @returns
     */
    async pause() {
        return new Promise(async (resolve) => {
            var _a, _b, _c, _d;
            if (!this.duplex) {
                this.logger.warn(`monitor to ${(_a = this.port) === null || _a === void 0 ? void 0 : _a.address} using ${(_b = this.port) === null || _b === void 0 ? void 0 : _b.protocol} already stopped`);
                return resolve();
            }
            // It's enough to close the connection with the client
            // to stop the monitor process
            this.duplex.end();
            this.logger.info(`stopped monitor to ${(_c = this.port) === null || _c === void 0 ? void 0 : _c.address} using ${(_d = this.port) === null || _d === void 0 ? void 0 : _d.protocol}`);
            this.duplex.on('end', resolve);
        });
    }
    /**
     * Stop the monitor currently running
     */
    async stop() {
        return this.pause().finally(this.stopMessagesHandlers.bind(this));
    }
    /**
     * Send a message to the running monitor, a well behaved monitor
     * will then send that message to the board.
     * We MUST NEVER send a message that wasn't a user's input to the board.
     * @param message string sent to running monitor
     * @returns a status to verify message has been sent.
     */
    async send(message) {
        if (!this.duplex) {
            return protocol_1.Status.NOT_CONNECTED;
        }
        const coreClient = await this.coreClient;
        const { instance } = coreClient;
        const req = new monitor_pb_1.MonitorRequest();
        req.setInstance(instance);
        req.setTxData(new TextEncoder().encode(message));
        return new Promise((resolve) => {
            var _a;
            if (this.duplex) {
                (_a = this.duplex) === null || _a === void 0 ? void 0 : _a.write(req, () => {
                    resolve(protocol_1.Status.OK);
                });
                return;
            }
            this.stop().then(() => resolve(protocol_1.Status.NOT_CONNECTED));
        });
    }
    /**
     *
     * @returns map of current monitor settings
     */
    async currentSettings() {
        await this.initialized;
        return this.settings;
    }
    // TODO: move this into MonitorSettingsProvider
    /**
     * Returns the possible configurations used to connect a monitor
     * to the board specified by fqbn using the specified protocol
     * @param protocol the protocol of the monitor we want get settings for
     * @param fqbn the fqbn of the board we want to monitor
     * @returns a map of all the settings supported by the monitor
     */
    async portMonitorSettings(protocol, fqbn) {
        const coreClient = await this.coreClient;
        const { client, instance } = coreClient;
        const req = new monitor_pb_1.EnumerateMonitorPortSettingsRequest();
        req.setInstance(instance);
        req.setPortProtocol(protocol);
        req.setFqbn(fqbn);
        const res = await new Promise((resolve, reject) => {
            client.enumerateMonitorPortSettings(req, (err, resp) => {
                if (!!err) {
                    reject(err);
                }
                resolve(resp);
            });
        });
        const settings = {};
        for (const iterator of res.getSettingsList()) {
            settings[iterator.getSettingId()] = {
                id: iterator.getSettingId(),
                label: iterator.getLabel(),
                type: iterator.getType(),
                values: iterator.getEnumValuesList(),
                selectedValue: iterator.getValue(),
            };
        }
        return settings;
    }
    /**
     * Set monitor settings, if there is a running monitor they'll be sent
     * to it, otherwise they'll be used when starting one.
     * Only values in settings parameter will be change, other values won't
     * be changed in any way.
     * @param settings map of monitor settings to change
     * @returns a status to verify settings have been sent.
     */
    async changeSettings(settings) {
        const config = new monitor_pb_1.MonitorPortConfiguration();
        const { pluggableMonitorSettings } = settings;
        const reconciledSettings = await this.monitorSettingsProvider.setSettings(this.monitorID, pluggableMonitorSettings || {});
        if (reconciledSettings) {
            for (const id in reconciledSettings) {
                const s = new monitor_pb_1.MonitorPortSetting();
                s.setSettingId(id);
                s.setValue(reconciledSettings[id].selectedValue);
                config.addSettings(s);
            }
        }
        this.updateClientsSettings({
            monitorUISettings: Object.assign(Object.assign({}, settings.monitorUISettings), { connected: !!this.duplex, serialPort: this.port.address }),
            pluggableMonitorSettings: reconciledSettings,
        });
        if (!this.duplex) {
            return protocol_1.Status.NOT_CONNECTED;
        }
        const coreClient = await this.coreClient;
        const { instance } = coreClient;
        const req = new monitor_pb_1.MonitorRequest();
        req.setInstance(instance);
        req.setPortConfiguration(config);
        this.duplex.write(req);
        return protocol_1.Status.OK;
    }
    /**
     * Starts the necessary handlers to send and receive
     * messages to and from the frontend and the running monitor
     */
    startMessagesHandlers() {
        if (!this.flushMessagesInterval) {
            const flushMessagesToFrontend = () => {
                if (this.messages.length) {
                    this.webSocketProvider.sendMessage(JSON.stringify(this.messages));
                    this.messages = [];
                }
            };
            this.flushMessagesInterval = setInterval(flushMessagesToFrontend, 32);
        }
        if (!this.onMessageReceived) {
            this.onMessageReceived = this.webSocketProvider.onMessageReceived((msg) => {
                const message = JSON.parse(msg);
                switch (message.command) {
                    case protocol_1.Monitor.ClientCommand.SEND_MESSAGE:
                        this.send(message.data);
                        break;
                    case protocol_1.Monitor.ClientCommand.CHANGE_SETTINGS:
                        this.changeSettings(message.data);
                        break;
                }
            });
        }
    }
    updateClientsSettings(settings) {
        this.settings = Object.assign(Object.assign({}, this.settings), settings);
        const command = {
            command: protocol_1.Monitor.MiddlewareCommand.ON_SETTINGS_DID_CHANGE,
            data: settings,
        };
        this.webSocketProvider.sendMessage(JSON.stringify(command));
    }
    /**
     * Stops the necessary handlers to send and receive messages to
     * and from the frontend and the running monitor
     */
    stopMessagesHandlers() {
        if (this.flushMessagesInterval) {
            clearInterval(this.flushMessagesInterval);
            this.flushMessagesInterval = undefined;
        }
        if (this.onMessageReceived) {
            this.onMessageReceived.dispose();
            this.onMessageReceived = undefined;
        }
    }
};
__decorate([
    (0, inversify_1.inject)(core_1.ILogger),
    (0, inversify_1.named)(exports.MonitorServiceName),
    __metadata("design:type", Object)
], MonitorService.prototype, "logger", void 0);
__decorate([
    (0, inversify_1.inject)(monitor_settings_provider_1.MonitorSettingsProvider),
    __metadata("design:type", Object)
], MonitorService.prototype, "monitorSettingsProvider", void 0);
__decorate([
    (0, inversify_1.inject)(web_socket_provider_1.WebSocketProvider),
    __metadata("design:type", Object)
], MonitorService.prototype, "webSocketProvider", void 0);
__decorate([
    (0, inversify_1.postConstruct)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], MonitorService.prototype, "init", null);
MonitorService = __decorate([
    __param(0, (0, inversify_1.inject)(monitor_service_factory_1.MonitorServiceFactoryOptions)),
    __metadata("design:paramtypes", [Object])
], MonitorService);
exports.MonitorService = MonitorService;
/**
 * Splits a string into an array without removing newline char.
 * @param s string to split into lines
 * @returns an lines array
 */
function splitLines(s) {
    return s.split(/(?<=\n)/);
}
//# sourceMappingURL=monitor-service.js.map