"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.SketchesServiceImpl = void 0;
const inversify_1 = require("@theia/core/shared/inversify");
const fs = require("fs");
const os = require("os");
const temp = require("temp");
const path = require("path");
const crypto = require("crypto");
const ncp_1 = require("ncp");
const util_1 = require("util");
const uri_1 = require("@theia/core/lib/common/uri");
const node_1 = require("@theia/core/lib/node");
const config_service_impl_1 = require("./config-service-impl");
const sketches_service_1 = require("../common/protocol/sketches-service");
const notification_service_server_1 = require("./notification-service-server");
const env_variables_1 = require("@theia/core/lib/common/env-variables");
const core_client_provider_1 = require("./core-client-provider");
const commands_pb_1 = require("./cli-protocol/cc/arduino/cli/commands/v1/commands_pb");
const decorators_1 = require("../common/decorators");
const glob = require("glob");
const promise_util_1 = require("@theia/core/lib/common/promise-util");
const service_error_1 = require("./service-error");
const is_temp_sketch_1 = require("./is-temp-sketch");
let SketchesServiceImpl = class SketchesServiceImpl extends core_client_provider_1.CoreClientAware {
    constructor() {
        super(...arguments);
        this.sketchSuffixIndex = 1;
    }
    async getSketches({ uri, exclude, }) {
        const [/*old,*/ _new] = await Promise.all([
            // this.getSketchesOld({ uri, exclude }),
            this.getSketchesNew({ uri, exclude }),
        ]);
        return _new;
    }
    async getSketchesNew({ uri, exclude, }) {
        const root = await this.root(uri);
        const pathToAllSketchFiles = await new Promise((resolve, reject) => {
            glob('/!(libraries|hardware)/**/*.{ino,pde}', { root }, (error, results) => {
                if (error) {
                    reject(error);
                }
                else {
                    resolve(results);
                }
            });
        });
        // Sort by path length to filter out nested sketches, such as the `Nested_folder` inside the `Folder` sketch.
        //
        // `directories#user`
        // |
        // +--Folder
        //    |
        //    +--Folder.ino
        //    |
        //    +--Nested_folder
        //       |
        //       +--Nested_folder.ino
        pathToAllSketchFiles.sort((left, right) => left.length - right.length);
        const container = sketches_service_1.SketchContainer.create(uri ? path.basename(root) : 'Sketchbook');
        const getOrCreateChildContainer = (parent, segments) => {
            if (segments.length === 1) {
                throw new Error(`Expected at least two segments relative path: ['ExampleSketchName', 'ExampleSketchName.{ino,pde}]. Was: ${segments}`);
            }
            if (segments.length === 2) {
                return parent;
            }
            const label = segments[0];
            const existingSketch = parent.sketches.find((sketch) => sketch.name === label);
            if (existingSketch) {
                // If the container has a sketch with the same label, it cannot have a child container.
                // See above example about how to ignore nested sketches.
                return undefined;
            }
            let child = parent.children.find((child) => child.label === label);
            if (!child) {
                child = sketches_service_1.SketchContainer.create(label);
                parent.children.push(child);
            }
            return child;
        };
        for (const pathToSketchFile of pathToAllSketchFiles) {
            const relative = path.relative(root, pathToSketchFile);
            if (!relative) {
                console.warn(`Could not determine relative sketch path from the root <${root}> to the sketch <${pathToSketchFile}>. Skipping. Relative path was: ${relative}`);
                continue;
            }
            const segments = relative.split(path.sep);
            if (segments.length < 2) {
                // folder name, and sketch name.
                console.warn(`Expected at least one segment relative path from the root <${root}> to the sketch <${pathToSketchFile}>. Skipping. Segments were: ${segments}.`);
                continue;
            }
            // the folder name and the sketch name must match. For example, `Foo/foo.ino` is invalid.
            // drop the folder name from the sketch name, if `.ino` or `.pde` remains, it's valid
            const sketchName = segments[segments.length - 2];
            const sketchFilename = segments[segments.length - 1];
            const sketchFileExtension = segments[segments.length - 1].replace(new RegExp(sketchName), '');
            if (sketchFileExtension !== '.ino' && sketchFileExtension !== '.pde') {
                console.warn(`Mismatching sketch file <${sketchFilename}> and sketch folder name <${sketchName}>. Skipping`);
                continue;
            }
            const child = getOrCreateChildContainer(container, segments);
            if (child) {
                child.sketches.push({
                    name: sketchName,
                    uri: node_1.FileUri.create(pathToSketchFile).toString(),
                });
            }
        }
        return container;
    }
    async root(uri) {
        return node_1.FileUri.fsPath(uri !== null && uri !== void 0 ? uri : (await this.sketchbookUri()));
    }
    async sketchbookUri() {
        const { sketchDirUri } = await this.configService.getConfiguration();
        return sketchDirUri;
    }
    async loadSketch(uri) {
        const { client, instance } = await this.coreClient;
        const req = new commands_pb_1.LoadSketchRequest();
        const requestSketchPath = node_1.FileUri.fsPath(uri);
        req.setSketchPath(requestSketchPath);
        req.setInstance(instance);
        const stat = new promise_util_1.Deferred();
        fs.lstat(requestSketchPath, (err, result) => err ? stat.resolve(err) : stat.resolve(result));
        const sketch = await new Promise((resolve, reject) => {
            client.loadSketch(req, async (err, resp) => {
                var _a;
                if (err) {
                    reject(isNotFoundError(err)
                        ? sketches_service_1.SketchesError.NotFound(fixErrorMessage(err, requestSketchPath, (_a = this.configService.cliConfiguration) === null || _a === void 0 ? void 0 : _a.directories.user), uri)
                        : err);
                    return;
                }
                const responseSketchPath = (0, is_temp_sketch_1.maybeNormalizeDrive)(resp.getLocationPath());
                if (requestSketchPath !== responseSketchPath) {
                    console.warn(`Warning! The request sketch path was different than the response sketch path from the CLI. This could be a potential bug. Request: <${requestSketchPath}>, response: <${responseSketchPath}>.`);
                }
                const resolvedStat = await stat.promise;
                if (resolvedStat instanceof Error) {
                    console.error(`The CLI could load the sketch from ${requestSketchPath}, but stating the folder has failed.`);
                    reject(resolvedStat);
                    return;
                }
                const { mtimeMs } = resolvedStat;
                resolve({
                    name: path.basename(responseSketchPath),
                    uri: node_1.FileUri.create(responseSketchPath).toString(),
                    mainFileUri: node_1.FileUri.create(resp.getMainFile()).toString(),
                    otherSketchFileUris: resp
                        .getOtherSketchFilesList()
                        .map((p) => node_1.FileUri.create(p).toString()),
                    additionalFileUris: resp
                        .getAdditionalFilesList()
                        .map((p) => node_1.FileUri.create(p).toString()),
                    rootFolderFileUris: resp
                        .getRootFolderFilesList()
                        .map((p) => node_1.FileUri.create(p).toString()),
                    mtimeMs,
                });
            });
        });
        return sketch;
    }
    async maybeLoadSketch(uri) {
        return this._isSketchFolder(uri);
    }
    get recentSketchesFsPath() {
        return this.envVariableServer
            .getConfigDirUri()
            .then((uri) => path.join(node_1.FileUri.fsPath(uri), 'recent-sketches.json'));
    }
    async loadRecentSketches(fsPath) {
        let data = {};
        try {
            const raw = await (0, util_1.promisify)(fs.readFile)(fsPath, {
                encoding: 'utf8',
            });
            data = JSON.parse(raw);
        }
        catch (_a) { }
        return data;
    }
    async markAsRecentlyOpened(uri) {
        let sketch = undefined;
        try {
            sketch = await this.loadSketch(uri);
        }
        catch (_a) {
            return;
        }
        if (await this.isTemp(sketch)) {
            return;
        }
        const fsPath = await this.recentSketchesFsPath;
        const data = await this.loadRecentSketches(fsPath);
        const now = Date.now();
        data[sketch.uri] = now;
        let toDeleteUri = undefined;
        if (Object.keys(data).length > 10) {
            let min = Number.MAX_SAFE_INTEGER;
            for (const uri of Object.keys(data)) {
                if (min > data[uri]) {
                    min = data[uri];
                    toDeleteUri = uri;
                }
            }
        }
        if (toDeleteUri) {
            delete data[toDeleteUri];
        }
        await (0, util_1.promisify)(fs.writeFile)(fsPath, JSON.stringify(data, null, 2));
        this.recentlyOpenedSketches().then((sketches) => this.notificationService.notifyRecentSketchesDidChange({ sketches }));
    }
    async recentlyOpenedSketches() {
        const configDirUri = await this.envVariableServer.getConfigDirUri();
        const fsPath = path.join(node_1.FileUri.fsPath(configDirUri), 'recent-sketches.json');
        let data = {};
        try {
            const raw = await (0, util_1.promisify)(fs.readFile)(fsPath, {
                encoding: 'utf8',
            });
            data = JSON.parse(raw);
        }
        catch (_a) { }
        const sketches = [];
        for (const uri of Object.keys(data).sort((left, right) => data[right] - data[left])) {
            try {
                const sketch = await this.loadSketch(uri);
                sketches.push(sketch);
            }
            catch (_b) { }
        }
        return sketches;
    }
    async cloneExample(uri) {
        const sketch = await this.loadSketch(uri);
        const parentPath = await this.createTempFolder();
        const destinationUri = node_1.FileUri.create(path.join(parentPath, sketch.name)).toString();
        const copiedSketchUri = await this.copy(sketch, { destinationUri });
        return this.loadSketch(copiedSketchUri);
    }
    async createNewSketch() {
        const monthNames = [
            'jan',
            'feb',
            'mar',
            'apr',
            'may',
            'jun',
            'jul',
            'aug',
            'sep',
            'oct',
            'nov',
            'dec',
        ];
        const today = new Date();
        const parentPath = await this.createTempFolder();
        const sketchBaseName = `sketch_${monthNames[today.getMonth()]}${today.getDate()}`;
        const config = await this.configService.getConfiguration();
        const sketchbookPath = node_1.FileUri.fsPath(config.sketchDirUri);
        let sketchName;
        // If it's another day, reset the count of sketches created today
        if (this.lastSketchBaseName !== sketchBaseName)
            this.sketchSuffixIndex = 1;
        let nameFound = false;
        while (!nameFound) {
            const sketchNameCandidate = `${sketchBaseName}${sketchIndexToLetters(this.sketchSuffixIndex++)}`;
            // Note: we check the future destination folder (`directories.user`) for name collision and not the temp folder!
            const sketchExists = await (0, util_1.promisify)(fs.exists)(path.join(sketchbookPath, sketchNameCandidate));
            if (!sketchExists) {
                nameFound = true;
                sketchName = sketchNameCandidate;
            }
        }
        if (!sketchName) {
            throw new Error('Cannot create a unique sketch name');
        }
        this.lastSketchBaseName = sketchBaseName;
        const sketchDir = path.join(parentPath, sketchName);
        const sketchFile = path.join(sketchDir, `${sketchName}.ino`);
        await (0, util_1.promisify)(fs.mkdir)(sketchDir, { recursive: true });
        await (0, util_1.promisify)(fs.writeFile)(sketchFile, `void setup() {
  // put your setup code here, to run once:

}

void loop() {
  // put your main code here, to run repeatedly:

}
`, { encoding: 'utf8' });
        return this.loadSketch(node_1.FileUri.create(sketchDir).toString());
    }
    /**
     * Creates a temp folder and returns with a promise that resolves with the canonicalized absolute pathname of the newly created temp folder.
     * This method ensures that the file-system path pointing to the new temp directory is fully resolved.
     * For example, on Windows, instead of getting an [8.3 filename](https://en.wikipedia.org/wiki/8.3_filename), callers will get a fully resolved path.
     * `C:\\Users\\KITTAA~1\\AppData\\Local\\Temp\\.arduinoIDE-unsaved2022615-21100-iahybb.yyvh\\sketch_jul15a` will be `C:\\Users\\kittaakos\\AppData\\Local\\Temp\\.arduinoIDE-unsaved2022615-21100-iahybb.yyvh\\sketch_jul15a`
     */
    createTempFolder() {
        return new Promise((resolve, reject) => {
            temp.mkdir({ prefix: is_temp_sketch_1.TempSketchPrefix }, (createError, dirPath) => {
                if (createError) {
                    reject(createError);
                    return;
                }
                fs.realpath.native(dirPath, (resolveError, resolvedDirPath) => {
                    if (resolveError) {
                        reject(resolveError);
                        return;
                    }
                    resolve(resolvedDirPath);
                });
            });
        });
    }
    async getSketchFolder(uri) {
        if (!uri) {
            return undefined;
        }
        let currentUri = new uri_1.default(uri);
        while (currentUri && !currentUri.path.isRoot) {
            const sketch = await this._isSketchFolder(currentUri.toString());
            if (sketch) {
                return sketch;
            }
            currentUri = currentUri.parent;
        }
        return undefined;
    }
    async isSketchFolder(uri) {
        const sketch = await this._isSketchFolder(uri);
        return !!sketch;
    }
    async _isSketchFolder(uri) {
        try {
            const sketch = await this.loadSketch(uri);
            return sketch;
        }
        catch (err) {
            if (sketches_service_1.SketchesError.NotFound.is(err)) {
                return undefined;
            }
            throw err;
        }
    }
    async isTemp(sketch) {
        return this.isTempSketch.is(node_1.FileUri.fsPath(sketch.uri));
    }
    async copy(sketch, { destinationUri }) {
        const source = node_1.FileUri.fsPath(sketch.uri);
        const exists = await (0, util_1.promisify)(fs.exists)(source);
        if (!exists) {
            throw new Error(`Sketch does not exist: ${sketch}`);
        }
        // Nothing to do when source and destination are the same.
        if (sketch.uri === destinationUri) {
            await this.loadSketch(sketch.uri); // Sanity check.
            return sketch.uri;
        }
        const copy = async (sourcePath, destinationPath) => {
            return new Promise((resolve, reject) => {
                ncp_1.ncp.ncp(sourcePath, destinationPath, async (error) => {
                    if (error) {
                        reject(error);
                        return;
                    }
                    const newName = path.basename(destinationPath);
                    try {
                        const oldPath = path.join(destinationPath, new uri_1.default(sketch.mainFileUri).path.base);
                        const newPath = path.join(destinationPath, `${newName}.ino`);
                        if (oldPath !== newPath) {
                            await (0, util_1.promisify)(fs.rename)(oldPath, newPath);
                        }
                        await this.loadSketch(node_1.FileUri.create(destinationPath).toString()); // Sanity check.
                        resolve();
                    }
                    catch (e) {
                        reject(e);
                    }
                });
            });
        };
        // https://github.com/arduino/arduino-ide/issues/65
        // When copying `/path/to/sketchbook/sketch_A` to `/path/to/sketchbook/sketch_A/anything` on a non-POSIX filesystem,
        // `ncp` makes a recursion and copies the folders over and over again. In such cases, we copy the source into a temp folder,
        // then move it to the desired destination.
        const destination = node_1.FileUri.fsPath(destinationUri);
        let tempDestination = await this.createTempFolder();
        tempDestination = path.join(tempDestination, sketch.name);
        await fs.promises.mkdir(tempDestination, { recursive: true });
        await copy(source, tempDestination);
        await copy(tempDestination, destination);
        return node_1.FileUri.create(destination).toString();
    }
    async archive(sketch, destinationUri) {
        await this.loadSketch(sketch.uri); // sanity check
        const { client } = await this.coreClient;
        const archivePath = node_1.FileUri.fsPath(destinationUri);
        // The CLI cannot override existing archives, so we have to wipe it manually: https://github.com/arduino/arduino-cli/issues/1160
        if (await (0, util_1.promisify)(fs.exists)(archivePath)) {
            await (0, util_1.promisify)(fs.unlink)(archivePath);
        }
        const req = new commands_pb_1.ArchiveSketchRequest();
        req.setSketchPath(node_1.FileUri.fsPath(sketch.uri));
        req.setArchivePath(archivePath);
        await new Promise((resolve, reject) => {
            client.archiveSketch(req, (err) => {
                if (err) {
                    reject(err);
                    return;
                }
                resolve(destinationUri);
            });
        });
        return destinationUri;
    }
    async getIdeTempFolderUri(sketch) {
        const genBuildPath = await this.getIdeTempFolderPath(sketch);
        return node_1.FileUri.create(genBuildPath).toString();
    }
    async getIdeTempFolderPath(sketch) {
        const sketchPath = node_1.FileUri.fsPath(sketch.uri);
        await fs.promises.readdir(sketchPath); // Validates the sketch folder and rejects if not accessible.
        const suffix = crypto.createHash('md5').update(sketchPath).digest('hex');
        return path.join(os.tmpdir(), `arduino-ide2-${suffix}`);
    }
    notifyDeleteSketch(sketch) {
        const sketchPath = node_1.FileUri.fsPath(sketch.uri);
        fs.rm(sketchPath, { recursive: true, maxRetries: 5 }, (error) => {
            if (error) {
                console.error(`Failed to delete sketch at ${sketchPath}.`, error);
            }
            else {
                console.error(`Successfully delete sketch at ${sketchPath}.`);
            }
        });
    }
};
__decorate([
    (0, inversify_1.inject)(config_service_impl_1.ConfigServiceImpl),
    __metadata("design:type", config_service_impl_1.ConfigServiceImpl)
], SketchesServiceImpl.prototype, "configService", void 0);
__decorate([
    (0, inversify_1.inject)(notification_service_server_1.NotificationServiceServerImpl),
    __metadata("design:type", notification_service_server_1.NotificationServiceServerImpl)
], SketchesServiceImpl.prototype, "notificationService", void 0);
__decorate([
    (0, inversify_1.inject)(env_variables_1.EnvVariablesServer),
    __metadata("design:type", Object)
], SketchesServiceImpl.prototype, "envVariableServer", void 0);
__decorate([
    (0, inversify_1.inject)(is_temp_sketch_1.IsTempSketch),
    __metadata("design:type", is_temp_sketch_1.IsTempSketch)
], SketchesServiceImpl.prototype, "isTempSketch", void 0);
__decorate([
    (0, decorators_1.duration)(),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], SketchesServiceImpl.prototype, "getSketchesNew", null);
SketchesServiceImpl = __decorate([
    (0, inversify_1.injectable)()
], SketchesServiceImpl);
exports.SketchesServiceImpl = SketchesServiceImpl;
// https://github.com/arduino/arduino-cli/issues/1797
function fixErrorMessage(err, sketchPath, sketchbookPath) {
    if (!sketchbookPath) {
        return err.details; // No way to repair the error message. The current sketchbook path is not available.
    }
    // Original: `Can't open sketch: no valid sketch found in /Users/a.kitta/Documents/Arduino: missing /Users/a.kitta/Documents/Arduino/Arduino.ino`
    // Fixed: `Can't open sketch: no valid sketch found in /Users/a.kitta/Documents/Arduino: missing $sketchPath`
    const message = err.details;
    const incorrectMessageSuffix = path.join(sketchbookPath, 'Arduino.ino');
    if (message.startsWith("Can't open sketch: no valid sketch found in") &&
        message.endsWith(`${incorrectMessageSuffix}`)) {
        const sketchName = path.basename(sketchPath);
        const correctMessagePrefix = message.substring(0, message.length - incorrectMessageSuffix.length);
        return `${correctMessagePrefix}${path.join(sketchPath, `${sketchName}.ino`)}`;
    }
    return err.details;
}
function isNotFoundError(err) {
    return service_error_1.ServiceError.is(err) && err.code === 5; // `NOT_FOUND` https://grpc.github.io/grpc/core/md_doc_statuscodes.html
}
/*
 * When a new sketch is created, add a suffix to distinguish it
 * from other new sketches I created today.
 * If 'sketch_jul8a' is already used, go with 'sketch_jul8b'.
 * If 'sketch_jul8b' already used, go with 'sketch_jul8c'.
 * When it reach 'sketch_jul8z', go with 'sketch_jul8aa',
 * and so on.
 */
function sketchIndexToLetters(num) {
    let out = '';
    let pow;
    do {
        pow = Math.floor(num / 26);
        const mod = num % 26;
        out = (mod ? String.fromCharCode(96 + mod) : (--pow, 'z')) + out;
        num = pow;
    } while (pow > 0);
    return out;
}
//# sourceMappingURL=sketches-service-impl.js.map