"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const monitor_settings_utils_1 = require("../../node/monitor-settings/monitor-settings-utils");
describe('longestPrefixMatch', () => {
    const settings = {
        'arduino:avr:uno-port1-protocol1': {
            name: 'Arduino Uno',
        },
        'arduino:avr:due-port1-protocol2': {
            name: 'Arduino Due',
        },
    };
    it('should return the exact prefix when found', async () => {
        const prefix = 'arduino:avr:uno-port1-protocol1';
        const { matchingPrefix } = (0, monitor_settings_utils_1.longestPrefixMatch)(prefix, settings);
        (0, chai_1.expect)(matchingPrefix).to.equal(prefix);
    });
    it('should return the exact object when the prefix match', async () => {
        const prefix = 'arduino:avr:uno-port1-protocol1';
        const { matchingSettings } = (0, monitor_settings_utils_1.longestPrefixMatch)(prefix, settings);
        (0, chai_1.expect)(matchingSettings).to.have.property('name').to.equal('Arduino Uno');
    });
    it('should return a partial matching prefix when a similar object is found', async () => {
        const prefix = 'arduino:avr:due-port2-protocol2';
        const { matchingPrefix } = (0, monitor_settings_utils_1.longestPrefixMatch)(prefix, settings);
        (0, chai_1.expect)(matchingPrefix).to.equal('arduino:avr:due');
    });
    it('should return the closest object when the prefix partially match', async () => {
        const prefix = 'arduino:avr:uno-port1-protocol2';
        const { matchingSettings } = (0, monitor_settings_utils_1.longestPrefixMatch)(prefix, settings);
        (0, chai_1.expect)(matchingSettings).to.have.property('name').to.equal('Arduino Uno');
    });
    it('should return an empty matching prefix when no similar object is found', async () => {
        const prefix = 'arduino:avr:tre-port2-protocol2';
        const { matchingPrefix } = (0, monitor_settings_utils_1.longestPrefixMatch)(prefix, settings);
        (0, chai_1.expect)(matchingPrefix).to.equal('');
    });
    it('should return an empty object when no similar object is found', async () => {
        const prefix = 'arduino:avr:tre-port1-protocol2';
        const { matchingSettings } = (0, monitor_settings_utils_1.longestPrefixMatch)(prefix, settings);
        (0, chai_1.expect)(matchingSettings).to.be.empty;
    });
});
describe('reconcileSettings', () => {
    const defaultSettings = {
        setting1: {
            id: 'setting1',
            label: 'Label setting1',
            type: 'enum',
            values: ['a', 'b', 'c'],
            selectedValue: 'b',
        },
        setting2: {
            id: 'setting2',
            label: 'Label setting2',
            type: 'enum',
            values: ['a', 'b', 'c'],
            selectedValue: 'b',
        },
        setting3: {
            id: 'setting3',
            label: 'Label setting3',
            type: 'enum',
            values: ['a', 'b', 'c'],
            selectedValue: 'b',
        },
    };
    it('should return default settings if new settings are missing', async () => {
        const newSettings = {};
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, defaultSettings);
        (0, chai_1.expect)(reconciledSettings).to.deep.equal(defaultSettings);
    });
    it('should add missing attributes copying it from the default settings', async () => {
        const newSettings = JSON.parse(JSON.stringify(defaultSettings));
        delete newSettings.setting2;
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, defaultSettings);
        (0, chai_1.expect)(reconciledSettings).to.have.property('setting2');
    });
    it('should remove wrong settings attributes using the default settings as a reference', async () => {
        const newSettings = JSON.parse(JSON.stringify(defaultSettings));
        newSettings['setting4'] = defaultSettings.setting3;
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, defaultSettings);
        (0, chai_1.expect)(reconciledSettings).not.to.have.property('setting4');
    });
    it('should reset non-value fields to those defiend in the default settings', async () => {
        const newSettings = JSON.parse(JSON.stringify(defaultSettings));
        newSettings['setting2'].id = 'fake id';
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, defaultSettings);
        (0, chai_1.expect)(reconciledSettings.setting2)
            .to.have.property('id')
            .equal('setting2');
    });
    it('should accept a selectedValue if it is a valid one', async () => {
        const newSettings = JSON.parse(JSON.stringify(defaultSettings));
        newSettings.setting2.selectedValue = 'c';
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, defaultSettings);
        (0, chai_1.expect)(reconciledSettings.setting2)
            .to.have.property('selectedValue')
            .to.equal('c');
    });
    it('should fall a back to the first valid setting when the selectedValue is not valid', async () => {
        const newSettings = JSON.parse(JSON.stringify(defaultSettings));
        newSettings.setting2.selectedValue = 'z';
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, defaultSettings);
        (0, chai_1.expect)(reconciledSettings.setting2)
            .to.have.property('selectedValue')
            .to.equal('a');
    });
    it('should accept any value if default values are not set', async () => {
        const wrongDefaults = JSON.parse(JSON.stringify(defaultSettings));
        wrongDefaults.setting2.values = [];
        const newSettings = JSON.parse(JSON.stringify(wrongDefaults));
        newSettings.setting2.selectedValue = 'z';
        const reconciledSettings = (0, monitor_settings_utils_1.reconcileSettings)(newSettings, wrongDefaults);
        (0, chai_1.expect)(reconciledSettings.setting2)
            .to.have.property('selectedValue')
            .to.equal('z');
    });
});
//# sourceMappingURL=monitor-settings-utils.test.js.map